/*
 * point.h
 *
 * This is an example of a classic point class in C++
 *
 * The implementation includes operator overloading, which is crucial if you are
 * working with any math classes.  We have inlined all of the operators for performance.
 * This is standard, unless you are working with something really complex like a matrix.
 *
 * Walker M. White
 * February 6, 2015
 */

// Prevent cyclical includes
#ifndef __POINT__H_
#define __POINT__H_

#include <sstream>

// Like a Java package.  Allows us to not use std:: in front of everything.
using namespace std;

/**
 * This is the class declaration.
 *
 * It looks like an interface in Java.  It has no implementations for any of the
 * methods. However, there are significant differences, like the fact that it
 * also contains the field declarations.
 */
class Point {
public:		// Unlike Java, these are expressed as a group
	/** The x-coordinate */
	float x;
	/** The y-coordinate */
	float y;
	/** The z-coordinate */
	float z;

	/**
	 * Create a new point object
	 *
	 * Note the use of default parameters.
	 *
	 * @param x  The initial x-coordinate
	 * @param y  The initial y-coordinate
	 * @param z  The initial z-coordinate
	 */
	Point(float x = 0.0f, float y = 0.0f, float z = 0.0f);

	/**
	 * Create a new point object
	 *
	 * This is copy constructor.  If you want to return a point from a function
	 * without allocating to the heap, you will need a copy constructor.  This
	 * means that the point is copied.  But if the data is small enough (it is
	 * just three floats) that may not be a big deal.
	 *
	 * @param p  The point to copy
	 * @param y  The initial y-coordinate
	 * @param z  The initial z-coordinate
	 */
	Point(const Point& p);

	/**
	 * Delete a point object
	 *
	 * This is ONLY necessary if the object contains other array, objects, or
	 * assets that it must delete once it is deleted.
	 */
	 ~Point();

	 /**
	  * Returns the string representation of this point.
	  *
	  * The value returned is obtained from sstream.  Because we are returning
	  * the value and it is not on the heap, this will COPY the contents of the
	  * stringstream to the next stack, essentially building the string twice.
	  *
	  * @return the string representation of this point.
	  */
	string toString();

	/**
	 * Returns the distance from this point to other
	 *
	 * Note the Point&.  If you do not do this, you are COPYING the point whenever
	 * you call this function.  We do not want to copy this point, so we pass by
	 * reference.
	 *
	 * @param other  Reference to point to measure distance to
	 *
	 * @return the distance from this point to other
	 */
	float dist(const Point& other);

	// OPERATOR OVERLOADING.  These are classic to in-line

	/**
	 *  Returns this point, with elements set to rhs point.
	 *
	 *  @param rhs The point to copy.
	 *
	 *  @return This point, with elements set to rhs point.
	 */
	Point& operator=(const Point &rhs) {
		x = rhs.x; y = rhs.y; z = rhs.z;
		return *this;
	}

	/**
	 *  Returns a point that is the sum of this and rhs.
	 *
	 *  @param rhs The point to add.
	 *
	 *  @return A point that is the sum of this and rhs.
	 */
	Point operator+(const Point &rhs) const {
		return (Point(*this)+=rhs);
	}

	/**
	 *  Returns a point that is the difference of this and rhs.
	 *
	 *  @param rhs The point to subtract.
	 *
	 *  @return A point that is the difference of this and rhs.
	 */
	Point operator-(const Point &rhs) const {
		return (Point(*this)-=rhs);
	}

	/**
	 *  Returns a point that is the product of this and the scalar rhs.
	 *
	 *  This method only supports multiplication by a scalar on the
	 *  right.  If we want to support multiplication on the left,
	 *  we will need another method.
	*
	 *  @param rhs The scalar to multiply by.
	 *
	 *  @return A point that is the product of this and the scalar rhs.
	 */
	Point operator*(const float &rhs) const {
		return (Point(*this)*=rhs);
	}

	/**
	 *  Returns this point, with elements added in place from rhs.
	 *
	 *  @param rhs The point to add in place.
	 *
	 *  @return This point, with elements added in place from rhs.
	 */
	Point& operator+=(const Point &rhs) {
		x += rhs.x; y += rhs.y; z += rhs.z;
		return *this;
	}

	/**
	 *  Returns this point, with elements subtracted in place by rhs.
	 *
	 *  @param rhs The point to subtract in place.
	 *
	 *  @return This point, with elements subtracted in place by rhs.
	 */
	 Point& operator-=(const Point &rhs) {
		x -= rhs.x; y -= rhs.y; z -= rhs.z;
		return *this;
	 }

	/**
	 *  Returns this point, multiplied on the left by the scalar rhs.
	 *
	 *  This method only supports multiplication by a scalar on the
	 *  right.  If we want to support multiplication on the left,
	 *  we will need another method.
	 *
	 *  @param rhs The scalar to multiply by.
	 *
	 *  @return This point, multiplied on the left by the scalar rhs.
	 */
	Point& operator*=(float rhs) {
		x *= rhs; y *= rhs; z *= rhs;
		return *this;
	}

	// FRIEND OPERATIONS (These are NOT methods, but allow access to private vars)
	/**
	 *  Returns a point that is the product of this and the scalar lhs.
	 *
	 *  This function only supports multiplication by a scalar on the
	 *  left.  If we want to support multiplication on the right, we
	 *  use the normal method in the class.
	*
	 *  @param rhs The scalar to multiply by.
	 *
	 *  @return A point that is the product of this and the scalar rhs.
	 */
	friend Point operator* (float lhs, const Point& p) {
		return p*lhs;
	}

}; // DO NOT FORGET THE SEMICOLON!!!

#endif
