/*
 * memory.cpp
 * 
 * This is a simple example to show how pointers and memory allocation work in C++.
 * Note that reference variables and pointers can go back and forth between each other.
 * Therefore the ONLY reason to use a pointer is when you need to dynamically allocate
 * an object on the heap.  You typically only need to do this when the amount of memory 
 * is either large or variable in size.  In other words, objects and arrays. 
 *
 * Walker M. White
 * February 6, 2016
 */
#include <stdio.h>   // We will not use cout in this example.

/**
 * Increments the contents of a pointer by 1.
 * 
 * Note that we use dereferencing (*variable) to access the contents of the variable.
 * If we executed x=x+1, we are changing the local variable x which is lost when the
 * call frame disappears.
 *
 * @param x  pointer to a int
 */
void incr_ptr(int* x) {
	*x = *x + 1;
}

/**
 * Increments the contents of a variable by 1.
 * 
 * Note that parameter x is a reference variable.  This means it is the same variable
 * as the one used in the function call, and it is not lost when the call frame appears.
 * That is why executing x = x+1 is not meaningless.
 *
 * @param x  reference to an int variable
 */
void incr_ref(int& x) {
	x = x + 1;
}

/**
 * Function to test out pointers and reference variables with ints.
 *
 * The purpose of this test is to show how pointers and references are interchangeable.
 */
void test_ints() {
	printf("Testing ints\n");
	int  x = 10; // Standard local variable
	int* y = new int(20); // Allocate to the heap
	
	printf("  x is now %d\n",x);
	printf("  y is now %d\n",*y);
	printf("\n");

	incr_ref(x);
	incr_ptr(y);

	printf("  x is now %d\n",x);
	printf("  y is now %d\n",*y);
	printf("\n");

	incr_ref(*y);
	incr_ptr(&x);
	
	printf("  x is now %d\n",x);
	printf("  y is now %d\n",*y);
	
	// MUST DELETE ANYTHING YOU CREATED!
	delete y;
	printf("Completed ints\n");
}

/**
 * Function to test out array memory allocation.
 *
 * The purpose of this test is to show WHY you might want new. While arrays can be
 * created by local variables without the use if new, you cannot RETURN a locally
 * created array.
 *
 * @param size  The array size
 */
int* test_array(int size) {
	printf("Testing array\n");
	//int array[size]; 			// OKAY FOR LOCAL USE ONLY
	int* array = new int[size];	// OKAY FOR RETURNING
	
	// Initialize
	for(int ii = 0; ii < size; ii++) {
		array[ii] = ii;	
	}
	
	// Increment like before
	for(int ii = 0; ii < size; ii++) {
		incr_ref(array[ii]);
		printf("  Array position %d is now %d\n",ii,array[ii]);
		incr_ptr(&(array[ii]));
		printf("  Array position %d is now %d\n",ii,array[ii]);
	}
	
	// We will leave deletion to the variable storing return value
	printf("Completed array\n");

	return array;
}


/**
 * Demonstrate pointers and memory allocation
 *
 * We have two tests, one with an int variable and one with an array of ints.  The 
 * int variables show how pointers and references are interchangeable.  Really, the
 * only time that you need to allocate memory is for something of large or variable 
 * length, like an array.
 */
int main() {
	printf("Starting main\n\n");
	
	test_ints();
	printf("\n");
	int* array = test_array(5);

	// DELETE THE ARRAY
	delete[] array;

	printf("\nFinished main\n");
}