/*
 * cornelltest.cpp
 *
 * This is the C++ equivalent to the cornelltest unit test module that we use in the
 * Python course. It has assert functions that will quit the application if they fail.
 * The error messages will also produce a stack trace, though the stack trace is C++
 * style, and may look very foreign to you.
 *
 * Unlike the other example code in this lecture, this module is VERY Unix specific.
 * The library "execinfo" is a GNU-specific library designed for Unix and OS X. I have
 * no idea if it works on Windows.
 *
 * Walker M. White
 * February 6, 2015
 */
#include <execinfo.h> // To get the stack trace
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sstream>

using namespace std;

// PRIVATE FUNCTIONS
#define EPSILON 0.000001f

/**
 * Quit the application with an error msg
 *
 * This function provides a C++-style stack trace when it quits. When printing out
 * the stack trace, it ignores this function and the one that immediately calling it.
 * That is because it is always called by asserts, and we want to stack trace to
 * begin with the function that called the assert
 *
 * @param message  The error message
 */
void quit_with_error(string message) {
	void *array[10];
	size_t size;
	
	// get void*'s for all entries on the stack
	size = backtrace(array, 10);
	
	// print out all the frames except the last two to
	fprintf(stderr, "Assertion Failed: %s\n", message.c_str());
	backtrace_symbols_fd(&(array[2]), size-2, STDERR_FILENO);
	exit(1);
}


// PUBLIC FUNCTIONS

/**
 * Quit if expected and received differ
 *
 * The meaning of "differ" for this function is !=.  As a result, this 
 * assert function is not necessarily reliable when expected and received 
 * are of type "float".  You should use the function assert_floats_equal 
 * for that application.
 *
 * @param expected  The answer we should have gotten
 * @param received  What we actually got
 */
void assert_equals(int expected,int received) {
	if (expected != received) {
		stringstream ss; // We have to use this because + does not work on strings
		ss << "expected " << expected << " but instead got " << received;
		quit_with_error(ss.str());
	}
}


/**
 * Quit if expected and received differ
 *
 * The meaning of "same" for this function is ==.  As a result, this 
 * assert function is not necessarily reliable when expected and received 
 * are of type "float".  You should use the function assert_not_floats_equal 
 * for that application.
 *
 * @param expected  The answer we do not want
 * @param received  What we actually got
 */
void assert_not_equals(int expected,int received) {
	if (expected == received) {
		stringstream ss; // We have to use this because + does not work on strings
		ss << "expected something different from " << expected;
		quit_with_error(ss.str());
	}
}

/**
 * Quit if received is False.
 * 
 * @param received  What we actually got
 */
void assert_true(bool received) {
	if (!received) {
		quit_with_error("expected true but instead got false");
	}
}

/**
 * Quit if received is True.
 * 
 * @param received  What we actually got
 */
void assert_false(bool received) {
	if (received) {
		quit_with_error("expected false but instead got true");
	}
}

/**
 * Quit if floats expected and received differ.
 *
 * Floats are compared to be within EPSILON of one another.  If they are farther
 * apart than this, this function produces an error.
 *
 * @param expected  The answer we should have gotten
 * @param received  What we actually got
 */
void assert_floats_equal(float expected, float received) {
	float diff = expected-received;
	if (diff > EPSILON || diff < -EPSILON) {
		stringstream ss; // We have to use this because + does not work on strings
		ss << "expected " << expected << " but instead got " << received;
		quit_with_error(ss.str());
	}
}


/**
 * Quit if floats expected and received are the same.
 *
 * Floats are compared to be within EPSILON of one another.  If they are within this
 * range, this function produces an error.
 *
 * @param expected  The answer we do not want
 * @param received  What we actually got
 */
void assert_floats_not_equal(float expected, float received) {
	float diff = expected-received;
	if (diff > EPSILON || diff < -EPSILON) {
		stringstream ss; // We have to use this because + does not work on strings
		ss << "expected something different from " << expected;
		quit_with_error(ss.str());
	}
}