/*
 * point.h
 *
 * This is an example of a classic point class in C++
 *
 * The header contains the class declaration, but usually does not contain any
 * implementations.  Note the important differences between Java and C++.
 *
 * Walker M. White
 * February 6, 2015
 */
 
// Prevent cyclical includes
#ifndef __POINT__H_ 
#define __POINT__H_

#include <sstream>

// Like a Java package.  Allows us to not use std:: in front of everything.
using namespace std;

/**
 * This is the class declaration.
 *
 * It looks like an interface in Java.  It has no implementations for any of the
 * methods. However, there are significant differences, like the fact that it 
 * also contains the field declarations.
 */
class Point {
private:	// Unlike Java, these are expressed as a group
	/** To format this point as a string */
	stringstream format;
		
public:		// Unlike Java, these are expressed as a group
	/** The x-coordinate */
	float x;
	/** The y-coordinate */
	float y;
	/** The z-coordinate */
	float z;
		
	/** 
	 * Create a new point object
	 *
	 * Note the use of default parameters.
	 *
	 * @param x  The initial x-coordinate
	 * @param y  The initial y-coordinate
	 * @param z  The initial z-coordinate
	 */
	Point(float x = 0.0f, float y = 0.0f, float z = 0.0f);
	
	/** 
	 * Create a new point object
	 *
	 * This is copy constructor.  If you want to return a point from a function
	 * without allocating to the heap, you will need a copy constructor.  This
	 * means that the point is copied.  But if the data is small enough (it is
	 * just three floats) that may not be a big deal.
	 *
	 * @param p  The point to copy
	 * @param y  The initial y-coordinate
	 * @param z  The initial z-coordinate
	 */
	Point(const Point& p);
	
	/**
	 * Delete a point object
	 *
	 * This is ONLY necessary if the object contains other array, objects, or 
	 * assets that it must delete once it is deleted.
	 */
	 ~Point();
	 
	 /**
	  * Returns the string representation of this point.
	  *
	  * The value returned is obtained from sstream.  Because we are returning
	  * the value and it is not on the heap, this will COPY the contents of the 
	  * stringstream to the next stack, essentially building the string twice.
	  *
	  * @return the string representation of this point.
	  */
	string toString();
	
    /**
     * Returns the distance from this point to other
     *
     * Note the Point&.  If you do not do this, you are COPYING the point whenever 
     * you call this function.  We do not want to copy this point, so we pass by
     * reference.
     *
     * @param other  Reference to point to measure distance to
     *
     * @return the distance from this point to other
     */
    float dist(const Point& other);

}; // DO NOT FORGET THE SEMICOLON!!!

#endif