/*
 * main.cpp
 * 
 * You do not need to know how to write templates, but you do need to know how to use 
 * them.  In particular, you need to understand the C++ standard template library.  It
 * is a lot like java.util in that it provides all of the container classes for C++.
 * It is also very weird.  In this example we show off a vector and a hashmap.
 *
 * Walker M. White
 * February 12, 2016
 */
#include <stdio.h>
#include <vector>
#include <string>
#include <unordered_map>
#include "a.h"


using namespace std;

/**
 * Tests out a vector from the Standard Template Library
 * 
 * A vector in C++ is the same as a vector in Java.  However, the methods are a little
 * different.  You access it like an array (because it has overridden []).
 *
 * More importantly, C++ iterators are weird.  A vector iterator overrides the dereference 
 * operator (*) so that it gives you the current element at the front of the iterator.
 *
 * This function shows how to add elements to a vector and how to use an iterator.
 *
 * @param size  the array size to make
 */
void test_vector(int size) {
	vector<int> array;  // Make a stack-based vector
	
	printf("Starting vector test\n");
	
	for(int ii = 0; ii < size; ii++) {
		array.push_back(ii*ii); // Add ii^2 to vector.
		printf("Vector size is now %d\n",(int)array.size());
	}
	
	int pos = size/2;
	printf("Position %d of the vector is %d\n",pos,array[pos]);
	array[pos] = -10*array[pos-1];
	printf("Position %d of the vector is %d\n",pos,array[pos]);
	
	
	// Now use an iterator to walk through the vector quickly
 	for (vector<int>::iterator it = array.begin() ; it != array.end(); ++it) {
 		printf("Next vector element is %d\n",*it);
 	}

	printf("Ending vector test\n");	
}

/**
 * Tests out a hashmap from the Standard Template Library
 * 
 * The actual name is unordered_map, but it is a hashmap.  The weird thing is that it
 * works like an array.  But instead of indexing by number, you index by key (because 
 * the class has overridden []).
 *
 * More importantly, C++ iterators are weird.  A hashmap iterator overrides the -> 
 * operator so that you can use it to access the key and value of the current element 
 * at the front of the iterator.
 *
 * This function shows how to add elements to a hashmap and how to use an iterator.
 */
void test_hashmap() {
	unordered_map<string,string> strmap; // Make a stack-based map

	printf("Starting hash map test\n");

	// Just initialize some stuff
	strmap["half"] = "baked";
	strmap["full"] = "moon";
	strmap["empty"] = "promise";
	strmap["meager"] = "portions";
	
	// Now use an iterator to walk through the hashmap quickly
 	for (unordered_map<string,string>::iterator it = strmap.begin() ; it != strmap.end(); ++it) {
 		printf("String \"%s\" maps to string \"%s\"\n", it->first.c_str(), it->second.c_str());
 	}
 	
 	strmap["empty"] = "calories";

 	// Now use an iterator to walk through the hashmap quickly
 	for (unordered_map<string,string>::iterator it = strmap.begin() ; it != strmap.end(); ++it) {
 		printf("String \"%s\" maps to string \"%s\"\n", it->first.c_str(), it->second.c_str());
 	}

	printf("Ending hash map test\n");	
}


/**
 * Tests out the custom template in a.h
 *
 * This function shows how to use a custom, user-defined template
 */
void test_a() {
	printf("Starting class A test\n");
	
	A<int> a1(3);
	printf("a1 output: %s\n", a1.toString().c_str());

	A<char> a2('x');
	printf("a2 output: %s\n", a2.toString().c_str());

	A< A<int>* > a3(&a1);
	printf("a3 output: %s\n", a3.toString().c_str());

	printf("Ending class A test\n");
}


/**
 * Demonstrate template containers
 *
 * In this example we test out a vector and a hashmap.
 */
int main() {
	printf("Starting main\n\n");
	
	test_vector(6);
	printf("\n");
	test_hashmap();
	printf("\n");
	test_a();

	printf("\nFinished main\n");
}