/*
 * point.h
 *
 * This is an alternative implementation of a point class in C++
 *
 * You will notice that this class declaration now contains some method implementations.
 * Those methods will now be in-lined when called.  As in-lining increases code size, 
 * you should limit it to getters and setters.
 *
 * Walker M. White
 * February 6, 2015
 */
 
// Prevent cyclical includes
#ifndef __POINT__H_ 
#define __POINT__H_

#include <sstream>

// Like a Java package.  Allows us to not use std:: in front of everything.
using namespace std;

/**
 * This is the class declaration.
 *
 * It looks like an interface in Java.  It has no implementations for any of the
 * methods. However, there are significant differences, like the fact that it 
 * also contains the field declarations.
 */
class Point {
private:	// Unlike Java, these are expressed as a group
	/** To format this point as a string */
	stringstream format;
	/** The x-coordinate */
	float x;
	/** The y-coordinate */
	float y;
	/** The z-coordinate */
	float z;
		
public:		// Unlike Java, these are expressed as a group
	/** 
	 * Create a new point object
	 *
	 * Note the use of default parameters.
	 *
	 * @param x  The initial x-coordinate
	 * @param y  The initial y-coordinate
	 * @param z  The initial z-coordinate
	 */
	Point(float x = 0.0f, float y = 0.0f, float z = 0.0f);
	
	/** 
	 * Create a new point object
	 *
	 * This is copy constructor.  If you want to return a point from a function
	 * without allocating to the heap, you will need a copy constructor.  This
	 * means that the point is copied.  But if the data is small enough (it is
	 * just three floats) that may not be a big deal.
	 *
	 * @param p  The point to copy
	 * @param y  The initial y-coordinate
	 * @param z  The initial z-coordinate
	 */
	Point(const Point& p);
	
	/**
	 * Delete a point object
	 *
	 * This is ONLY necessary if the object contains other array, objects, or 
	 * assets that it must delete once it is deleted.
	 */
	~Point();
	 
	// GETTERS AND SETTERS (Which are inlined)
	/**
	 * Returns the x-coordinate 
	 * 
	 * @return the x-coordinate
	 */
	float getX() const { return x; } 
	
	/**
	 * Sets the x-coordinate 
	 * 
	 * @param value  the x-coordinate
	 */
	void setX(float value) { x = value; }

	/**
	 * Returns the y-coordinate 
	 * 
	 * @return the y-coordinate
	 */
	float getY() const { return y; }

	/**
	 * Sets the y-coordinate 
	 * 
	 * @param value  the y-coordinate
	 */
	void setY(float value) { y = value; }

	/**
	 * Returns the z-coordinate 
	 * 
	 * @return the z-coordinate
	 */
	float getZ() const { return z; }

	/**
	 * Sets the z-coordinate 
	 * 
	 * @param value  the z-coordinate
	 */
	void setZ(float value) { z = value; }

	// NORMAL METHODS
	
	/**
	 * Returns the string representation of this point.
	 *
	 * The value returned is obtained from sstream.  Because we are returning
	 * the value and it is not on the heap, this will COPY the contents of the 
	 * stringstream to the next stack, essentially building the string twice.
	 *
	 * @return the string representation of this point.
	 */
	string toString();
	
    /**
     * Returns the distance from this point to other
     *
     * Note the Point&.  If you do not do this, you are COPYING the point whenever 
     * you call this function.  We do not want to copy this point, so we pass by
     * reference.
     *
     * @param other  Reference to point to measure distance to
     *
     * @return the distance from this point to other
     */
    float dist(const Point& other);

}; // DO NOT FORGET THE SEMICOLON!!!

#endif