/*
 * Copyright (c) 1998 by Interdisciplinary Center Herzliya
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE INTERDISCIPLINARY CENTER HERZLIYA BE LIABLE
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND
 * ITS DOCUMENTATION, EVEN IF THE INTERDISCIPLINARY CENTER HERZLIYA
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE INTERDISCIPLINARY CENTER HERZLIYA SPECIFICALLY DISCLAIMS ANY
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE 
 * SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE
 * INTERDISCIPLINARY CENTER HERZLIYA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 * */

// $Header: /usr/u/raoul/cvs/cs415-storage-server/il/ac/idc/storage/server/StorageUnit.java,v 1.1.1.1 1999/09/17 18:07:29 raoul Exp $ 

package il.ac.idc.storage.server;

import il.ac.idc.storage.*;
import java.io.*;
import java.util.*;
//
//
// StorageUnit
//
//

/** The base class for all (both) types of storage units. 

    @see DataStore
    @see DirectoryStore
  */
public abstract class StorageUnit {
  /** 
    * The server that manages this StorageUnit 
    */
  SimpleStorageServer server;
  /**
    * This StorageUnit's StorageID 
    */
  StorageID  sid;
  /**
    * The index file of this storage unit. This value could be recomputed
    * from the StorageID, but rather than constantly recreating the associated
    * File object, we cache its value here.
    */
  File indexFN;

  /** 
    * An int that is used when writing out the index file to indicate that
    * the associated StorageUnit is a Data Storage unit.
    */
  public static final int DataStore = 0;
  /** 
    * An int that is used when writing out the index file to indicate that
    * the associated StorageUnit is a directory Storage unit.
    */
  public static final int DirectoryStore = 1;
  /** 
    * An int that is stored in an index file to indicate that 
    * the associated StorageUnit has been deleted.
    */
  public static final int DeadStore = 255;

  /** Synchronize the "in memory" of the StorageUnit with what is persistent.
    * For Data storage units this means just saving the index file. For directories,
    * more will need to be done.
    *
    * @exception StorageException CatchAll for any sort of problem with the 
    *		  Shared Storage System.
    */
  abstract public void saveStorageUnit()
    throws StorageException;
  
  /** Restore the state of a StorageUnit from its persistent storage.
    * This is usually used when starting the storage sever.
    *
    * @exception StorageException CatchAll for any sort of problem with the 
    *		  Shared Storage System.
    */
  abstract public void restoreStorageUnit()
    throws StorageException ;

  /** Read som number of bytes from the persistent portion of the StorageUnit and 
    * return them as a byte array.
    *
    * @exception StorageException CatchAll for any sort of problem with the 
    *		  Shared Storage System.
    */
  abstract public byte[] read(long vad, long length)
    throws StorageException;
  
  /** Delete a Storage unit. Usually this means write a dead index file,
    * but the underlying storage on the disk might also be reclaimed.
    *
    * @exception StorageException CatchAll for any sort of problem with the 
    *		  Shared Storage System.
    */
  public void deleteUnit() 
  throws StorageException {
    try {
      DataOutputStream ostr 
	= new DataOutputStream(new FileOutputStream(indexFN));
      
      ostr.writeByte(DeadStore);
      ostr.close();
    }
    catch(IOException e) {
      StorageException.signalIOError(e);
    }
  }
}

