/*
 * Copyright (c) 1998 by Interdisciplinary Center Herzliya
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE INTERDISCIPLINARY CENTER HERZLIYA BE LIABLE
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND
 * ITS DOCUMENTATION, EVEN IF THE INTERDISCIPLINARY CENTER HERZLIYA
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE INTERDISCIPLINARY CENTER HERZLIYA SPECIFICALLY DISCLAIMS ANY
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE 
 * SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE
 * INTERDISCIPLINARY CENTER HERZLIYA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 * */

// $Header: /usr/u/raoul/cvs/cs415-storage-server/il/ac/idc/storage/server/SimpleStorageServer.java,v 1.1.1.1 1999/09/17 18:07:29 raoul Exp $ 

package il.ac.idc.storage.server;

import il.ac.idc.storage.*;

import java.lang.Thread;
import java.net.*;
import java.io.*;
import java.util.*;

/** Represents the storage system itself. Classes like server.ServerUDP
  * and server.ServerTCP (and others) serve as an interface between 
  * this class and the wire. 
  */
public class SimpleStorageServer implements StorageServer {
    /** Directory with the files for this storage server */
    protected File directory;   
    /** File with persistent Server Information */
    File definitionFile;
    Hashtable storageUnits;	    // Association between StorageID's and storage units
    StorageID nextSID;		    // Next data storage ID
    StorageID nextDSID;		    // Next directory storage ID
    DirectoryStore rootDirectory;	    // The root directory (SID = -1)

    /** Restores a storage server from a given directory.  If the directory 
     * is empty then StorgeServer file is created. Otherwise, the contents 
     * of the file are used to initialize the StorageServer. 
     *
     * @param dir A String that indicates where the data files are stored.
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public SimpleStorageServer(String dir) 
	throws StorageException {
	this(new File(dir));
    }

    /** Restores a storage server from a given directory.  If the directory 
     * is empty then StorgeServer file is created. Otherwise, the contents 
     * of the file are used to initialize the StorageServer. 

     * @param dir A File that indicates where the data files are stored.
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public SimpleStorageServer(File dir) 
	throws StorageException {
	directory = dir;
	storageUnits = new Hashtable();
	definitionFile = new File(directory, "StorageServer.def");

	Debug.println("StorageServer Directory: " + directory);

	try{
	    if (definitionFile.exists()) {
		try {
		    DataInputStream defStream 
			= new DataInputStream(new FileInputStream(definitionFile));
		    nextSID = new StorageID(defStream.readLong());
		    nextDSID = new StorageID(defStream.readLong());
		    defStream.close();
		}
		catch (IOException e) {
		    System.err.println("***Badly formed StorageServer.def file");
		    System.err.println("***Delete the file to reset the StorageID counters");
		}
	    } else {
		nextSID = StorageID.none;
		nextDSID = null;	      // Special case for creation of root directory
	
	
		rootDirectory  = getDirectoryStore(new StorageID(-1));
		if (null == rootDirectory) {
		    rootDirectory = getDirectoryStore(createDirectoryStore(new Credentials(0)));
		}
	Debug.println("Created root directory: " + rootDirectory.sid);

		syncStorageServer();
	    }
	    Debug.println("Last Data Store StorageID = " + nextSID);
	    Debug.println("Last Directory Store StorageID = " + nextDSID);
	}
	catch(SecurityException e) {
	    Debug.showError("Security Exception starting server", e);
	}
    }

    /** The temporary state of the StorageServer itself is written to disk.
     * This is just the definition file with the nextSID and nextDSID 
     * information.
     *
     * @exception StorageException CatchAll for errors in the StorageSystem 
     */
    private void syncStorageServer() 
	throws StorageException {
	try {
	    DataOutputStream defStream 
		= new DataOutputStream(new FileOutputStream(definitionFile));
      
	    defStream.writeLong(nextSID.toLong());
	    defStream.writeLong(nextDSID.toLong());
	    defStream.close();
	}
	catch (IOException e) {
	    StorageException.signalIOError(e);
	}
    }

    /** Compute the name of the index file from a StorageID and who 
     * our server is.

     @param id  A Storage ID for the index file that is to be generated.
    */
    public File computeIndexFile(StorageID id) {
	long idNum = id.toLong();
    
	if (idNum < 0) {
	    return new File(directory, -idNum + "dir.index");
	} else {
	    return new File(directory, idNum + ".index");
	}
    }
  
    /** Create a <EM>data</EM> storage unit on this Storage Server that can
     * contain at least size number of bytes of data.  
     * 
     * @param size the number of bytes in the storage unit that is created.
     * @return the StorageID of the newly created data storage unit
     * @exception StorageException When something unusual happens
     */
    public StorageID createDataStore(Credentials c, long size) 
	throws StorageException {
	nextSID = nextSID.next();
	StorageUnit su = new DataStore(this, nextSID);

	syncStorageServer();
	su.saveStorageUnit();

	storageUnits.put(nextSID, su);
	return nextSID;
    }

    /** Create a <EM>directory</EM> storage unit on this Storage Server that can
     * contain at least size number of bytes of data.  
     * 
     * @param size the number of bytes in the storage unit that is created.
     * @return the StorageID of the newly created data storage unit
     * @exception StorageException When something unusual happens
     */
    public StorageID createDirectoryStore(Credentials c) 
	throws StorageException {
	if (nextDSID == null) {
	    nextDSID = new StorageID(-1);
	} else {
	    nextDSID = nextDSID.next();
	}

	StorageUnit su = new DirectoryStore(this, nextDSID);

	syncStorageServer();
	su.saveStorageUnit();

	storageUnits.put(nextDSID, su);
	return nextDSID;
    }

    /**	Delete a storage unit. 
     *
     * @return an acknowledgement code.
     * @exception StorageException Deleted an nonexistent, or already deleted
     *				  storage unit.
     */
    public void deleteUnit(Credentials c, StorageID id)
	throws StorageException {
	StorageUnit su = (StorageUnit)storageUnits.get(id);

	if (su == null) {
	    throw new StorageException(StorageException.DeleteNExistSU);
	}
	return;
    }

    /** Get a storage unit from a StorageID. Loads the storage unit
     * from disk if necessary. This routine returns a DataStore.
     *
     * @param sid The StorageID that identifies the desired StorageUnit
     * @returns A DataStore corresponding to the indicated StorageID.
     * @exception StorageException Signaled if the sid corresponds to 
     *				  a Directory Store or is non-existent.
     */
    private DataStore getDataStore(StorageID sid) 
	throws StorageException {
	Object store = storageUnits.get(sid);

	if (store == null &&
	    computeIndexFile(sid).exists()) {
	    DataStore ds = new DataStore(this, sid);
      
	    ds.restoreStorageUnit();
	    storageUnits.put(sid, ds);
	    return ds;
	} else if (sid.isDataStore()) {
	    return (DataStore)store;
	} else {
	    throw new StorageException(StorageException.UnitMismatch);
	}
    }

    /** Get a storage unit from a StorageID. Loads the storage unit
     * from disk if necessary. This routine returns a DirectoryStore.
     *
     * @param sid The StorageID that identifies the desired StorageUnit
     * @returns A DirectoryStore corresponding to the indicated StorageID.
     * @exception StorageException Signaled if the sid corresponds to 
     *				  a Data Store or is non-existent.
     */
    private DirectoryStore getDirectoryStore(StorageID sid)
	throws StorageException {
	Object store = storageUnits.get(sid);

	if (store == null &&
	    computeIndexFile(sid).exists()) {
	    DirectoryStore ds = new DirectoryStore(this, sid);
      
	    ds.restoreStorageUnit();
	    storageUnits.put(sid, ds);
	    return ds;
	} else if (sid.isDirectoryStore()) {
	    return (DirectoryStore)store;
	}
	else {
	    throw new StorageException(StorageException.UnitMismatch);
	}
    }

    /** Given a StorageID, load the index file for the corresponding
     * storage unit, create the appropriate classes, and generally 
     * make this storage unit ready for use.
     *
     * It would have been more natural for this method to have been a 
     * static method of StorageUnit. Unfortunately, StorageUnit is an 
     * abstract class and you cannot include a static method in an abstract class!
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public StorageUnit restoreStorage(StorageID sid) 
	throws StorageException {
	File indFile = computeIndexFile(sid);
	DataInputStream istr;      
	StorageUnit su;

	try {
	    istr = new DataInputStream(new FileInputStream(indFile));
      
	    switch(istr.readByte()) {
	    case StorageUnit.DataStore:
		su = new DataStore(this, sid);
		break;
      
	    case StorageUnit.DirectoryStore:
	
		su = new DirectoryStore(this, sid);
		break;
      
	    default:
		throw new StorageException(StorageException.NotStorageID);
	    }
      
	    su.restoreStorageUnit();
	    storageUnits.put(sid, su);
	    return su;
	}
	catch (FileNotFoundException e) {
	    throw new StorageException(StorageException.IndexFNF);
	}
	catch(IOException e) {
	    StorageException.signalIOError(e);
	    return null;
	}
    }

    /** Given a StorageID return the corresponding storage unit.
     * If the storage unit is not currently cached, its index file
     * is loaded from disk.
     *
     * @param sid The StorageID of the desired storage unit.
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    private StorageUnit getStorageUnit(StorageID sid) 
	throws StorageException {
	Object store = storageUnits.get(sid);

	if (store == null &&
	    computeIndexFile(sid).exists()) {
	    return restoreStorage(sid);
	} else if (store instanceof StorageUnit) {
	    storageUnits.put(sid, store);
	    return (StorageUnit)store;
	} else {
	    throw new StorageException(StorageException.UnitMismatch);
	}
    }

    /** Save the index files to disk for all of the Storage Units
     * that are currently cached.
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public void syncIndexes() 
	throws StorageException {
	for (Enumeration e = storageUnits.elements(); e.hasMoreElements(); ) {
	    StorageUnit su = (StorageUnit)e.nextElement();

	    su.saveStorageUnit();
	}
    }


    /** Reads the data contained in the storage unit and returns it as an array
     * of bytes. The number of bytes read may be less than len (although it
     * won't be less than WriteData.DATA_LENGTH if length is greater than
     * WriteData.DATA_LENGTH). If an error occurs, an exception is thrown.
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public byte[] readData(Credentials c, StorageID sid, long vad, long length)
	throws StorageException {
	// note that this code does some unnecessary copying.
	int actual_length = (int)Math.min(length,WriteData.DATA_LENGTH);
	long current_address = vad;
 
	byte barray[];
	ByteArrayOutputStream bstream = new ByteArrayOutputStream(actual_length);
	StorageUnit su = getStorageUnit(sid);
 
	do {
	    barray = su.read(current_address, actual_length);
	    actual_length = actual_length - barray.length;
	    current_address = current_address + barray.length;
	    bstream.write(barray,0,barray.length);
	} while (actual_length > 0);
	// note that in the above, actualLength should never become negative.
 
	return bstream.toByteArray();
    }

    /** Writes the indicated data into the storage unit at the indicated
     * point.  The first len bytes in data are written. The value
     * returned is the return value of the acknowledgement. 
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public void writeData(Credentials cred, StorageID unitID, 
			  long start, long len, byte[] data) 
	throws StorageException { 
	getDataStore(unitID).write(start, data, 0, (int)len);
    };

    /** Read an attribute of the storage unit 
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public byte[] readAttribute(Credentials cred, StorageID unitID, int Attribute)
	throws StorageException {
	throw new StorageException(StorageException.Unimplemented);
    }

    /** Writes an attribute of the storage unit. The return value is the
     * return value of the acknowledgement. 
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public int writeAttribute(Credentials cred, StorageID unitID, int Attribute, byte[] data)
	throws StorageException {
	throw new StorageException(StorageException.Unimplemented);
    }
	
    /** Lookups up a string in a directory and returns the StorageID of 
     * the corresponding StorageUnit. 
     *
     * @param cred The Credentials of the originator of the query
     * @param directory The StorageID of the directory to be searched
     * @param unitName The String to be looked up in the directory
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public StorageID lookup(Credentials cred, StorageID directory, String unitName)
	throws StorageException {
	return ((DirectoryStore)getStorageUnit(directory)).lookup(unitName);
    }

    /** Binds the storage unit to a string name in the directory storage
     * unit.  
     *
     * @param cred The Credentials of the originator of the query
     * @param directory The StorageID of the directory to be searched
     * @param unitName The String to be looked up in the directory
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public void bind(Credentials cred, StorageID directory, String unitName, StorageID unit)
	throws StorageException {
	((DirectoryStore)getStorageUnit(directory)).bind(unitName, unit);
    }
		
    /** Unbinds a string name in the directory storage unit. 
     *
     * @param cred The Credentials of the originator of the query
     * @param directory The StorageID of the directory to be searched
     * @param unitName The String to be looked up in the directory
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public void unbind(Credentials cred, StorageID directory, String unitName)
	throws StorageException {
	((DirectoryStore)getStorageUnit(directory)).unbind(unitName);
    }

    /** Locks the region of the storage unit for exclusive use by the
     * current client. Any negative acknowledgement is indicated by 
     * throwing a StorageException
     *
     * @param cred The Credentials of the originator of the query
     * @param unit The StorageID of the Storage Unit to be locked
     * @param start The first byte to be locked 
     * @param length The number of bytes to be locked
     * @exception StorageException Indicates a negative acknowledgement
     */
    public void lock(Credentials cred, StorageID unit, long start, long length) 
	throws StorageException {
	throw new StorageException(StorageException.Unimplemented);
    }
		
    /** Unlocks the region of the storage unit for exclusive use by the
     * current client. Any negative acknowledgement is indicated by 
     * throwing a StorageException
     *
     * @param cred The Credentials of the originator of the query
     * @param unit The StorageID of the Storage Unit to be unlocked
     * @param start The first byte to be locked 
     * @param length The number of bytes to be unlocked
     * @exception StorageException Indicates a negative acknowledgement
     */
    public void unlock(Credentials cred, StorageID unit, long start, long length)
	throws StorageException {
	throw new StorageException(StorageException.Unimplemented);
    }
}

