/*
 * Copyright (c) 1998 by Interdisciplinary Center Herzliya
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE INTERDISCIPLINARY CENTER HERZLIYA BE LIABLE
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND
 * ITS DOCUMENTATION, EVEN IF THE INTERDISCIPLINARY CENTER HERZLIYA
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE INTERDISCIPLINARY CENTER HERZLIYA SPECIFICALLY DISCLAIMS ANY
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE 
 * SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE
 * INTERDISCIPLINARY CENTER HERZLIYA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 * */

// $Header: /usr/u/raoul/cvs/cs415-storage-server/il/ac/idc/storage/server/ServerUDP.java,v 1.1.1.1 1999/09/17 18:07:29 raoul Exp $ 

package il.ac.idc.storage.server;

import il.ac.idc.storage.*;
import java.net.*;
import java.io.*;

public class ServerUDP {
    static SimpleStorageServer sss;

    /** The one and only argument to this method is the directory of
     * data store.  To start an instance of the UDP storage server on 
     * a machine, switch to the SYP\lib directory and type
     * jview ServerUDP <EM> dir </EM> where <EM> dir </EM> is the directory
     * in which the servers files should be kept. 
     */
    public static void main (String[] args) {
        DatagramPacket incoming;
        Message packet;
        File servDir = null;

        if (args.length == 1) {
            servDir = new File(args[0]);
        } else if (args.length == 3 && args[0].equals("-p") 
		   && args[1].equals("ServerUDP")) {
            // If we are running inside the DeveloperStudio environment, extra
            // arguments are passed to Java view 
            servDir = new File(args[2]);
        } else {
            System.err.println("Improper number of arguments (" 
			       + args.length + "): ServerUDP dir");
            for (int i = 0; i < args.length; i++) {
                System.err.println("args[" + i + "] = " + args[i]);
            }
            Exit();
        }

        if (!servDir.exists()) {
            System.err.println("Directory (" + servDir + ") doesn't exist");
            Exit();
        }
        if (!servDir.isDirectory()) {
            System.err.println("File (" + servDir + ") is not a directory");
            Exit();
        }

        try {
            sss = new SimpleStorageServer(servDir);

            DatagramSocket ds = new DatagramSocket(Message.UDPPort);
            byte[] buffer = new byte[Message.UDPMessageSize];
            // If an error occurs in the previous lines, we have a serious
            // problem.  Print an error message and terminate the process.

            System.out.println("Shared Storge Server 0.1 running");

      
            while(true) {
                try {
                    // If there is an error in this loop, throw the packet away,
                    // print an error message, and continue processing other 
                    // packets.
                    incoming = new DatagramPacket(buffer, buffer.length);
                    ds.receive(incoming);
                    packet = Message.parseMessage(incoming);

                    Debug.println(1,"Received packet: " + packet);

                    DatagramPacket outgoing;
                    Message resp = response(packet);
                    if (resp == null) {
                        System.err.println("No response provided for message");
                    } else {
                        byte[] data = resp.toByteArray();
                        Message m = Message.parseMessage(data);
                        Debug.println(2,"Buffer msg: " + m.toString());
                        Debug.println(2,"Sending data: " + Hexify.toHexString(data));
                        outgoing = 
                            new DatagramPacket(data, data.length, 
                                               incoming.getAddress(), incoming.getPort());
                        ds.send(outgoing);
                        Debug.println(1,"  Sent packet: " + resp);
                    }
                }
                catch (IOException e) {
                    System.err.println(e);
                    e.printStackTrace(System.err);
                }
            }
        }
        catch (SocketException e) {
            System.err.println(e);
        }
        catch (StorageException e) {
            System.err.println(e);
        }
    }  
  
    static void Exit() {
        try {
            System.in.read();
        } 
        catch (IOException e) {
            System.err.println(e);
        }
        System.exit(-1);
    }

    /*  This is the way we would like to implement the response methods
        but it seems the dynamic dispatching in Java isn't quite as good 
        as we'd like.  See below for how it is really implemented.

        static Message response(Create packet) {
        try {
        StorageID newID = sss.createDataStore(packet.cred, packet.size);
        packet.id = newID;          // Reuse the old packet!!
        return packet;
        }
        catch (StorageException e) {
        return new Acknowledge(e.getErrorCode());
        }
        }

        static Message response(CreateDirectory packet) {
        try {
        StorageID newID = sss.createDirectoryStore(packet.cred);

        packet.id = newID;
        return packet;
        }    
        catch (StorageException e) {
        return new Acknowledge(e.getErrorCode());
        }
        }

        static Message response(Delete packet) {
        try {
        sss.deleteUnit(packet.cred, packet.id);
        return new Acknowledge(0);
        }
        catch (StorageException e) {
        return new Acknowledge(e.getErrorCode());
        }
        }
  
        static Message response(ReadData packet) {
        try {
        byte[] buffer 
        = sss.readData(packet.cred, packet.id, packet.offset, packet.length);

        return new WriteData(packet.cred, packet.id, packet.offset, packet.length,
        buffer);
        }
        catch (StorageException e) {
        return new Acknowledge(e.getErrorCode());
        }
        }

        static Message response(WriteData packet) {
        try {
        sss.writeData(packet.cred, packet.id, packet.offset, packet.length,
        packet.data);
        return new Acknowledge(0);
        }
        catch (StorageException e) {
        return new Acknowledge(e.getErrorCode());
        }
        }

        static Message response(Message packet) {
        return new Acknowledge(StorageException.Unimplemented);
        }
    */

    static Message response(Message packet) {
        try {
            switch(packet.marker) {
            case Message.Create: {
                Create pack = (Create)packet;

                StorageID newID = sss.createDataStore(pack.cred, pack.size);
                pack.id = newID;          // Reuse the old packet!!
                return packet;
            }

            case Message.CreateDir: {
                CreateDirectory pack = (CreateDirectory)packet;

                StorageID newID = sss.createDirectoryStore(pack.cred);
                pack.id = newID;
                return packet;
            }

            case Message.Delete: {
                Delete pack = (Delete)packet;
        
                sss.deleteUnit(pack.cred, pack.id);
                return new Acknowledge(0,pack.uniq);
            }

            case Message.ReadData: {
                ReadData pack = (ReadData)packet;

                byte[] buffer 
                    = sss.readData(pack.cred, pack.id, pack.offset, pack.length);

                return new WriteData(pack.cred, pack.id, pack.offset, buffer.length,
                                     buffer,pack.uniq);
            }

            case Message.WriteData: {
                WriteData pack = (WriteData)packet;

                sss.writeData(pack.cred, pack.id, pack.offset, pack.length,
                              pack.data);
                return new Acknowledge(0,pack.uniq);
            }

            case Message.Lookup: {
                Lookup pack = (Lookup)packet;

                StorageID id = sss.lookup(pack.cred, pack.dir,pack.name);
                if (id == null) id = StorageID.none;

                return new Bind(pack.cred, pack.dir, id, pack.name, pack.uniq);
            }

            case Message.Bind: {
                Bind pack = (Bind)packet;

                sss.bind(pack.cred, pack.dir, pack.name, pack.id);
                return new Acknowledge(0,pack.uniq);
            }

            case Message.Unbind: {
                Unbind pack = (Unbind)packet;

                sss.unbind(pack.cred, pack.dir, pack.name);
                return new Acknowledge(0,pack.uniq);
            }

            default:    
                return new Acknowledge(StorageException.Unimplemented,packet.uniq);
            }
        }
        catch (StorageException e) {
	    Debug.println(0,"Storage Exception occured while parsing message.\n" + e);
            return new Acknowledge(e.getErrorCode(),packet.uniq);
        } catch (NullPointerException e) {
	    System.err.println("UNEXPECTED ERROR");
	    System.err.println(packet);
	    throw e;
	}
    }

}

