/*
 * Copyright (c) 1998 by Interdisciplinary Center Herzliya
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE INTERDISCIPLINARY CENTER HERZLIYA BE LIABLE
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND
 * ITS DOCUMENTATION, EVEN IF THE INTERDISCIPLINARY CENTER HERZLIYA
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE INTERDISCIPLINARY CENTER HERZLIYA SPECIFICALLY DISCLAIMS ANY
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE 
 * SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE
 * INTERDISCIPLINARY CENTER HERZLIYA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 * */

// $Header: /usr/u/raoul/cvs/cs415-storage-server/il/ac/idc/storage/server/DirectoryStore.java,v 1.1.1.1 1999/09/17 18:07:29 raoul Exp $ 

package il.ac.idc.storage.server;

import il.ac.idc.storage.*;
import java.io.*;
import java.util.*;

/** A StorageUnit that holds an association between String's and StorageID's.
  * This is usually used to represent a file system or some other naming 
  * structure in a storage system.
  */
public class DirectoryStore extends StorageUnit {
  File indexFN;
  Hashtable direct;

  /** Reconstruct a Directory Storage Unit from the data stored on disk .
    *
    * @exception StorageException CatchAll for errors in the StorageSystem
    */
  public DirectoryStore(SimpleStorageServer s, StorageID storID) 
  throws StorageException {
    server = s;
    sid = storID;
    indexFN = new File(server.directory, -(storID.toLong()) + "dir.index");
    direct = new Hashtable();

    saveStorageUnit();
  }

  public StorageID lookup(String unitName) {
    return (StorageID)direct.get(unitName);
  }
  
  public void bind(String unitName, StorageID unit) {
    direct.put(unitName, unit);
    saveStorageUnit();
  }

  public void unbind(String unitName) {
    direct.remove(unitName);
    saveStorageUnit();
  }
  
  /** Saves all of the in-memory information about the storage unit to disk so
      that it can be restored by a restoreStorageUnit command. 
    *
    * @exception StorageException CatchAll for errors in the StorageSystem
    */
  public void saveStorageUnit() 
  throws StorageException { 
    try {
      DataOutputStream file 
	= new DataOutputStream(new FileOutputStream(indexFN));
      // The first byte written indicates whether this is a data storage unit
      // or a directory storage unit.
      file.writeByte(DirectoryStore);
      
      // Indicate how many elements there are in the directory
      file.writeLong(direct.size());
      
      for (Enumeration enum = direct.keys(); enum.hasMoreElements(); ) {
	String name = (String)enum.nextElement();
	StorageID sid = (StorageID)direct.get(name);
	
	file.writeUTF(name);
	file.writeLong(sid.toLong());
      }
    }
    catch (IOException e) {
      throw new StorageException(StorageException.IOError);
    }
  }

  /** Reconstruct a Storage Unit from the data stored on Disk 
    *
    * @exception StorageException CatchAll for errors in the StorageSystem
    */
  public void restoreStorageUnit() 
  throws StorageException {
    try {
      DataInputStream file
	= new DataInputStream(new FileInputStream(indexFN));
      byte DirStorKey = file.readByte();
      // Number of entries in Directory
      long nEntries = file.readLong();
      
      if (DirStorKey != DirectoryStore)
	throw new StorageException(StorageException.UnitMismatch);
      
      // Clear the directory, so we can reload it.
      direct.clear();
      for (int i = 0; i < nEntries; i++) {
	String name = file.readUTF();
	StorageID sid = new StorageID(file.readLong());
	
	direct.put(name, sid);
      }
    }
    catch (IOException e) {
      throw new StorageException(StorageException.IOError);
    }
  }

  public byte[] read(long vad, long length) {
    return new byte[1];
  }


}

