/*
 * Copyright (c) 1998 by Interdisciplinary Center Herzliya
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE INTERDISCIPLINARY CENTER HERZLIYA BE LIABLE
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND
 * ITS DOCUMENTATION, EVEN IF THE INTERDISCIPLINARY CENTER HERZLIYA
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE INTERDISCIPLINARY CENTER HERZLIYA SPECIFICALLY DISCLAIMS ANY
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE 
 * SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE
 * INTERDISCIPLINARY CENTER HERZLIYA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 * */

// $Header: /usr/u/raoul/cvs/cs415-storage-server/il/ac/idc/storage/client/ThreadsafeUDPStorageServer.java,v 1.1.1.1 1999/09/17 18:07:29 raoul Exp $ 

package il.ac.idc.storage.client;

import il.ac.idc.storage.*;
import java.net.*;
import java.io.*;
import java.util.*;

/** Implements the StorageServer interface, as a client, communicating
 * through a UDP channel.
 *
 * This is a threadsafe implementation of the UDPStorageServer class.  i.e., a
 * number of threads can simultaneously use this interface over a lossy
 * network.  It does not, however, tolerate packet losses.  Adding this would
 * require adding a server-side cache for non-idempotent operations.  
 *
 */
public class ThreadsafeUDPStorageServer implements StorageServer {
    private InetAddress server;
    private DatagramSocket socket;
    private Random rgen;
    private RequestQueue rqueue;
    private ReceiveThread rthread;
    private Object sendsocket;

    /** Create a reference to a StorageServer, and cache a copy of the
     * user's credentials 
     *
     * @param address The internet address of the server.
     * @exception IOException Some problem arose dealing with the underlying 
     *             file system.
     */
    public ThreadsafeUDPStorageServer(InetAddress address) 
        throws IOException {
        init(address);
    }

    public ThreadsafeUDPStorageServer (String serverName)
        throws IOException {
        try {
            init(InetAddress.getByName(serverName));
        } catch (UnknownHostException e) {
            System.out.println("Failed to connect to server.");
            System.exit(-1);
        }
    }

    private void init(InetAddress address) 
        throws IOException {
        // Must validate address, by making sure that the address refers
        // to a valid Shared Storage Server
        server = address;
        socket = new DatagramSocket();
        rgen = new Random();
        rqueue = new RequestQueue();
        sendsocket = new Object();

        rthread = new ReceiveThread(rqueue,socket,server);
        rthread.start();
    }
    

    /** Transmit a Message on the server's socket and parse the result.
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    private Message transmit(Message out) 
        throws StorageException {
        Message msg;
        try {
            UniqMessageHolder umh;
            byte[] data = out.toByteArray();

            Debug.println(1,"Sending: " + out);      

            // If the message size is larger than the packet size, throw an exception.
            if (data.length >= Message.UDPMessageSize) {
                throw new StorageException(StorageException.MessageNetworkIncompatible);
            }

            DatagramPacket dgp
                = new DatagramPacket(data, data.length, server, Message.UDPPort);
            
            umh = new UniqMessageHolder();
            umh.uniq = out.uniq;
            umh.msg = null;

            // Add the message to the queue and send the message -- in that order
            rqueue.put(umh.uniq, umh);
            // Are sockets threadsafe?

            synchronized (sendsocket) {
                socket.send(dgp);
            }


            // wait until the thread that receives messages does a notify() on
            // this object.
            
            synchronized(umh) {
                while (umh.msg == null) {
                    try {
                        umh.wait();
                    } catch (InterruptedException e) {}
                }
            }
            // once we wake up, the message should be stored.
            msg = umh.msg;
            if (msg == null) 
                throw new StorageException(StorageException.BadWireAPI);
            Debug.println(1,"  Received: " + msg);
            return msg;
        }
        catch(IOException e) {
            StorageException.signalIOError(e);
            return null;
        }
    }
     
    /** Create a <EM>data</EM> storage unit on this Storage Server that can
     * contain at least size number of bytes of data.  
     * 
     * @param size the number of bytes in the storage unit that is created.
     * @return the StorageID of the newly created data storage unit
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public StorageID createDataStore(Credentials cred, long size) 
        throws StorageException {
	int uniq;
	synchronized (rgen) {
	    uniq = rgen.nextInt();
	}
	
        Message response = 
            transmit(new Create(cred, StorageID.none, size, 
                                uniq));

        if (response.marker == Message.Create) {
            return ((Create)response).id;
        } else throw new StorageException(StorageException.CreateDataFail);
    }

    /** Create a <EM>directory</EM> storage unit on this Storage Server that can
     * contain at least size number of bytes of data.  
     * 
     * @param size the number of bytes in the storage unit that is created.
     * @return the StorageID of the newly created data storage unit
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public StorageID createDirectoryStore(Credentials cred) 
        throws StorageException {
	int uniq;
	synchronized (rgen) {
	    uniq = rgen.nextInt();
	}


        Message response 
            = transmit(new CreateDirectory(cred, StorageID.none, uniq));

        if (response.marker == Message.CreateDir) {
            return ((CreateDirectory)response).id;
        } else throw new StorageException(StorageException.CreateDirFail);
    }

    /** Delete a storage unit. 
     *
     * @return an acknowledgement code.
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public void deleteUnit(Credentials cred, StorageID id) 
        throws StorageException {
	int uniq;
	synchronized (rgen) {
	    uniq = rgen.nextInt();
	}

        Message response = transmit(new Delete(cred, id, uniq));

        expectAcknowledge(response);
    }

    /** Reads the data contained in the storage unit and returns it as an array
     * of bytes. The number of bytes read may be less than len (although it
     * won't be less than WriteData.DATA_LENGTH if length is greater than
     * WriteData.DATA_LENGTH). If an error occurs, an exception is thrown.
     *
     * @exception StorageException CatchAll for errors in the StorageSystem */
    public byte[] readData(Credentials cred, StorageID unitID, long start, long len) 
        throws StorageException {
	int uniq;
	synchronized (rgen) {
	    uniq = rgen.nextInt();
	}

        Message response = transmit(new ReadData(cred, unitID, start, len, 
						 uniq));
    
        if (response.marker == Message.WriteData) {
            return ((WriteData)response).data;
        } else {
            expectAcknowledge(response);
            return null;      // Should always thrown in previous statement
        }
    }

    /** Writes the indicated data into the storage unit at the indicated
     * point.  The first len bytes in data are written. The value
     * returned is the return value of the acknowledgement. 
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public void writeData(Credentials cred, StorageID unitID, long start, 
			  long len, byte[] data) 
        throws StorageException {
	int uniq;
	synchronized (rgen) {
	    uniq = rgen.nextInt();
	}

        Message response 
            = transmit(new WriteData(cred, unitID, start, len, data, uniq));

        expectAcknowledge(response);
    }

    /** Checks to see if the message is an an acknowledement message. 
     * If not, or if the acknowledgement code is not zero, signals 
     * StorageException.
     *
     * @exception StorageException Signaled whenever the message is not 
     *                             a positive acknowledgement
     */
    private void expectAcknowledge(Message mess) 
        throws StorageException {
        if (mess.marker != Message.Acknowledge) {
            System.err.println("Wrong Class message received: " + mess);
            throw new StorageException(StorageException.IOError);
        } else {
            Acknowledge ack = (Acknowledge)mess;

            if (ack.data != 0)
                throw new StorageException(ack.data);
        }
    }
 
    /** <bold> UNIMPLEMENTED </bold> Read an attribute of the storage unit 
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public byte[] readAttribute(Credentials cred, StorageID unitID, int Attribute)
        throws StorageException {
        return null;
    }

    /** <bold> UNIMPLEMENTED </bold> Writes an attribute of the storage
     * unit. The return value is the return value of the acknowledgement.
     *
     * @exception StorageException CatchAll for errors in the StorageSystem */
    public int writeAttribute(Credentials cred, StorageID unitID, int Attribute, byte[] data)
        throws StorageException {
        return -1;
    }
        
    /** Lookups up a string in a directory and returns the StorageID of 
     * the corresponding StorageUnit. 
     *
     * @param cred The Credentials of the originator of the query
     * @param directory The StorageID of the directory to be searched
     * @param unitName The String to be looked up in the directory
     * @exception StorageException CatchAll for errors in the StorageSystem
     */

    public StorageID lookup(Credentials c, StorageID directory, String unitName)
        throws StorageException {
	int uniq;
	synchronized (rgen) {
	    uniq = rgen.nextInt();
	}

        Message response = transmit(new Lookup(c, directory , unitName, uniq));

        if (response.marker == Message.Bind) {
            StorageID newID = ((Bind)response).id;

            if (newID.toLong() == 0) return null;
            else return newID;
        } else if (response.marker == Message.Acknowledge) {
            Acknowledge ack = (Acknowledge)response;

            throw new StorageException(ack.data);
        } else {
            throw new StorageException(-250);
        }
    }

    /** Binds the storage unit to a string name in the directory storage
     * unit.  
     *
     * @param cred The Credentials of the originator of the query
     * @param directory The StorageID of the directory to be searched
     * @param unitName The String to be looked up in the directory
     * @exception StorageException CatchAll for errors in the StorageSystem
     */

    public void bind(Credentials cred, StorageID directory, String unitName, 
                     StorageID unit)
        throws StorageException {
	int uniq;
	synchronized (rgen) {
	    uniq = rgen.nextInt();
	}


        Message response = transmit(new Bind(cred, directory , unit , unitName, uniq));

        if (response.marker != Message.Acknowledge) 
            throw new StorageException(StorageException.BindFailed);
    }
                
    /** Unbinds a string name in the directory storage unit. 
     *
     * @param cred The Credentials of the originator of the query
     * @param directory The StorageID of the directory to be searched
     * @param unitName The String to be looked up in the directory
     * @exception StorageException CatchAll for errors in the StorageSystem
     */

    public void unbind(Credentials c, StorageID directory, String unitName)
        throws StorageException {
	int uniq;
	synchronized (rgen) {
	    uniq = rgen.nextInt();
	}

        Message response = transmit(new Unbind(c, directory , unitName, uniq));

        if (response.marker != Message.Acknowledge) 
            throw new StorageException(StorageException.UnBindFailed);
    }

    /** <em> UNIMPLEMENTED </em> Locks the region of the storage unit for
     * exclusive use by the current client. Any negative acknowledgement is
     * indicated by throwing a StorageException
     *
     * @param cred The Credentials of the originator of the query
     * @param unit The StorageID of the Storage Unit to be locked
     * @param start The first byte to be locked 
     * @param length The number of bytes to be locked
     * @exception StorageException Indicates a negative acknowledgement */
    public void lock(Credentials c, StorageID unit, long start, long length) 
        throws StorageException {
    }
                
    /** <bold> UNIMPLEMENTED </bold> Unlocks the region of the storage unit for
     * exclusive use by the current client. Any negative acknowledgement is
     * indicated by throwing a StorageException
     *
     * @param cred The Credentials of the originator of the query
     * @param unit The StorageID of the Storage Unit to be unlocked
     * @param start The first byte to be locked 
     * @param length The number of bytes to be unlocked
     * @exception StorageException Indicates a negative acknowledgement */
    public void unlock(Credentials c, StorageID unit, long start, long length) 
        throws StorageException {
    }
}

class RequestQueue {
    /* Doesn't need to be synchrnoized because the Hashtable is synchronized */
    private Hashtable requests;

    public RequestQueue(){
        requests = new Hashtable();
    }

    public void put(int uniq, UniqMessageHolder umh) {
        requests.put(new Integer(uniq),umh);
    }

    // Takes a response along with an ID, and returns the previosly logged request that matches the ID
    public UniqMessageHolder get(int uniq) {
        return (UniqMessageHolder)requests.remove(new Integer(uniq));
    }
    
    public boolean containsKey(int uniq) {
        return requests.containsKey(new Integer(uniq));
    }
}



// Request queue: rqueue
class ReceiveThread extends Thread {
    DatagramSocket socket;
    byte[] buffer;
    DatagramPacket packet;
    UniqMessageHolder umh;
    Message msg;
    RequestQueue rqueue;

    public ReceiveThread(RequestQueue rqueue, DatagramSocket socket, InetAddress server) {
        this.socket = socket;
        this.rqueue = rqueue;
        buffer = new byte[Message.UDPMessageSize];
        packet = new DatagramPacket(buffer, buffer.length,
                                    server, Message.UDPPort);
    }

    public void run() {
        while (true){
            try{
                // It appears that reusing packets is a bad idea.  You can reuse the buffer, though....
                packet = new DatagramPacket(buffer, buffer.length);
                socket.receive(packet);
                byte[] buf = packet.getData();
                msg = Message.parseMessage(packet);
		//                Debug.println(3,"Raw message: " + msg.toString());
                if (rqueue.containsKey(msg.uniq)) {
                    // If there is a message waiting with the appropriate
                    // uniqifier, then get it, and, call notify on it 
                    umh = (UniqMessageHolder)rqueue.get(msg.uniq);
                    synchronized (umh) {
			umh.msg = msg;
                        umh.notify();
                    }
                } else {
		    Debug.println(0,"Message received with uniqifier " + msg.uniq +
				  ", but no message sent with that uniqifier."); 
		    Debug.println(1,"Message: " + msg);
                }
            } catch(IOException e) {
                Debug.showError(1,"IOException on message receive.", e);
                // there's really no one sensible to whom to pass this error on to, 
                // so we just print it out (assuming that debugging is enabled.)
            }
        }
    }
}

class UniqMessageHolder {
    public int uniq;
    public Message msg;
}
