/*
 * Copyright (c) 1998 by Interdisciplinary Center Herzliya
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE INTERDISCIPLINARY CENTER HERZLIYA BE LIABLE
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND
 * ITS DOCUMENTATION, EVEN IF THE INTERDISCIPLINARY CENTER HERZLIYA
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE INTERDISCIPLINARY CENTER HERZLIYA SPECIFICALLY DISCLAIMS ANY
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE 
 * SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE
 * INTERDISCIPLINARY CENTER HERZLIYA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 * */

// $Header: /usr/u/raoul/cvs/cs415-storage-server/il/ac/idc/storage/StorageServer.java,v 1.1.1.1 1999/09/17 18:07:27 raoul Exp $ 

package il.ac.idc.storage;

import java.net.*;
import java.io.*;

/** A class that represents the cleint's view of a Shared Storage Server. 
  * All Shared Storage Server objects must implement this interface.  At the 
  * moment on UDPStorageServer exists.

  * @see UDPStorageServer
*/
public interface StorageServer {
  /** Create a <EM>data</EM> storage unit on this Storage Server that can
    * contain at least size number of bytes of data.  
    * 
    * @param size the number of bytes in the storage unit that is created.
    * @return the StorageID of the newly created data storage unit
    * @exception StorageException Catch all for all storage system errors
    */
  public StorageID createDataStore(Credentials cred, long size) 
     throws StorageException;

  /** Create a <EM>directory</EM> storage unit on this Storage Server that can
    * contain at least size number of bytes of data.  
    * 
    * @param size the number of bytes in the storage unit that is created.
    * @return the StorageID of the newly created data storage unit
    * @exception StorageException Catch all for all storage system errors
    */
  public StorageID createDirectoryStore(Credentials cred) 
    throws StorageException;

  /**	Delete a storage unit. 
    *
    * @return an acknowledgement code.
    * @exception StorageException Catch all for all storage system errors
    */
  public void deleteUnit(Credentials cred, StorageID id) 
    throws StorageException;

  /** Reads the data contained in the storage unit and returns it as an array
    * of bytes. The number of bytes read may be less than len (although it
    * won't be less than WriteData.DATA_LENGTH if length is greater than
    * WriteData.DATA_LENGTH).  If an error occurs, an exception is thrown.
    *
    * @exception StorageException Catch all for all storage system errors */
  public byte[] readData(Credentials cred, StorageID unitID, long start, long len) 
    throws StorageException;

  /** Writes the indicated data into the storage unit at the indicated
    * point.  The first len bytes in data are written. The value
    * returned is the return value of the acknowledgement. 
    *
    * @exception StorageException Catch all for all storage system errors
    */
  public void writeData(Credentials cred, StorageID unitID, 
		        long start, long len, byte[] data) 
    throws StorageException;

  /** Executes cc's run() method on the server.  Returns the same byte
    * array as run().
    */
  public byte[] rexec(Credentials cred, ClientCode cc);

  /** Read an attribute of the storage unit 
    *
    * @exception StorageException Catch all for all storage system errors
    */
  public byte[] readAttribute(Credentials cred, StorageID unitID, int Attribute)
    throws StorageException ;

  /** Writes an attribute of the storage unit. The return value is the
    * return value of the acknowledgement. 
    *
    * @exception StorageException Catch all for all storage system errors
    */
  public int writeAttribute(Credentials cred, StorageID unitID, int Attribute, byte[] data)
    throws StorageException ;
	
  /** Lookups up a string in a directory and returns the StorageID of 
    * the corresponding StorageUnit. 
    *
    * @param cred The Credentials of the originator of the query
    * @param directory The StorageID of the directory to be searched
    * @param unitName The String to be looked up in the directory
    * @exception StorageException Catch all for all storage system errors
    */
  public StorageID lookup(Credentials cred, StorageID directory, String unitName)
    throws StorageException ;

  /** Binds the storage unit to a string name in the directory storage
    * unit.  
    *
    * @param cred The Credentials of the originator of the query
    * @param directory The StorageID of the directory to be searched
    * @param unitName The String to be looked up in the directory
    * @exception StorageException Catch all for all storage system errors
    */
  public void bind(Credentials cred, StorageID directory, String unitName, StorageID unit)
    throws StorageException ;
		
  /** Unbinds a string name in the directory storage unit. 
    *
    * @param cred The Credentials of the originator of the query
    * @param directory The StorageID of the directory to be searched
    * @param unitName The String to be looked up in the directory
    * @exception StorageException Catch all for all storage system errors
    */
  public void unbind(Credentials cred, StorageID directory, String unitName)
    throws StorageException ;

  /** Locks the region of the storage unit for exclusive use by the
    * current client. Any negative acknowledgement is indicated by 
    * throwing a StorageException Catch all for all storage system errors
    *
    * @param cred The Credentials of the originator of the query
    * @param unit The StorageID of the Storage Unit to be locked
    * @param start The first byte to be locked 
    * @param length The number of bytes to be locked
    * @exception StorageException Indicates a negative acknowledgement
    */
  public void lock(Credentials cred, StorageID unit, long start, long length) 
    throws StorageException;
		
  /** Unlocks the region of the storage unit for exclusive use by the
    * current client. Any negative acknowledgement is indicated by 
    * throwing a StorageException Catch all for all storage system errors
    *
    * @param cred The Credentials of the originator of the query
    * @param unit The StorageID of the Storage Unit to be unlocked
    * @param start The first byte to be locked 
    * @param length The number of bytes to be unlocked
    * @exception StorageException Indicates a negative acknowledgement
    */
  public void unlock(Credentials cred, StorageID unit, long start, long length) 
    throws StorageException;
}
