/*
 * Copyright (c) 1998 by Interdisciplinary Center Herzliya
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE INTERDISCIPLINARY CENTER HERZLIYA BE LIABLE
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND
 * ITS DOCUMENTATION, EVEN IF THE INTERDISCIPLINARY CENTER HERZLIYA
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE INTERDISCIPLINARY CENTER HERZLIYA SPECIFICALLY DISCLAIMS ANY
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE 
 * SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE
 * INTERDISCIPLINARY CENTER HERZLIYA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 * */

// $Header: /usr/u/raoul/cvs/cs415-storage-server/il/ac/idc/storage/Message.java,v 1.1.1.1 1999/09/17 18:07:27 raoul Exp $ 

package il.ac.idc.storage;

import java.io.*;
import java.net.DatagramPacket;
import java.util.*;


/** The base class for all message objects.  Within the storage
 *  server, all messages are represented by objects which are 
 *  subclassed from the Message class
 */
public abstract class Message {
   public final static int UDPPort = 2163;
   public final static int UDPMessageSize = 1000;
   /** specifies the message subtype.  This is only necessary because Java
    *  method dispatch is inefficient. */
   public int marker;
   /** A number for distinguishing messages belonging to different transactions */
   public int uniq;
   public Random rgen;

   public static final int ReadData    = ('R' << 24) + ('E' << 16) + ('A' << 8) + 'D';
   public static final int WriteData   = ('W' << 24) + ('R' << 16) + ('I' << 8) + 'T';
   public static final int Acknowledge = ('A' << 24) + ('C' << 16) + ('K' << 8) + 'N';
   public static final int Create      = ('C' << 24) + ('R' << 16) + ('E' << 8) + 'A';
   public static final int CreateDir   = ('C' << 24) + ('R' << 16) + ('D' << 8) + 'R';
   public static final int Delete      = ('D' << 24) + ('E' << 16) + ('T' << 8) + 'E';
   public static final int Lookup      = ('L' << 24) + ('O' << 16) + ('O' << 8) + 'K';
   public static final int Bind        = ('B' << 24) + ('I' << 16) + ('N' << 8) + 'D';
   public static final int Unbind      = ('U' << 24) + ('B' << 16) + ('N' << 8) + 'D';
   public static final int Lock        = ('L' << 24) + ('O' << 16) + ('C' << 8) + 'K';
   public static final int Unlock      = ('U' << 24) + ('N' << 16) + ('L' << 8) + 'K';

   // The port number to be used by the storage server.
   static final int StoragePort = 23123;

   /** Parses a DataInputStream to produce an instance of a class
    * that represents a message.
    *
    * @param istr A DataInputStream that contains the raw form of 
    *             the data for the packet.
    * @exception StorageException Some problem has arisen in parsing 
    *             the raw packet data. The stream terminated abnormally
    *             or was badly formatted.
    */
   static Message parseMessage(DataInputStream istr) 
      throws StorageException, IOException {
      int messageCode = istr.readInt();
      switch (messageCode) {
      case ReadData:
	 return new ReadData(istr);
      case WriteData:
	 return new WriteData(istr);
      case Acknowledge:
	 return new Acknowledge(istr);
      case Create:
	 return new Create(istr);
      case CreateDir:
	 return new CreateDirectory(istr);
      case Delete:
	 return new Delete(istr);
      case Lookup:
	 return new Lookup(istr);
      case Bind:
	 return new Bind(istr);
      case Unbind:
	 return new Unbind(istr);
      case Lock:
	 return new Lock(istr);
      case Unlock:
	 return new Unlock(istr);
      default:
	 throw new StorageException(StorageException.BadWireAPI);
      }
   }

   /** Parses a message that this contained in a byte array. 
    * Returns a Message object.
    * 
    * @exception StorageException Some error arose in parse the message
    */
   public static Message parseMessage(byte[] buffer) 
      throws StorageException {
   
      try {
	 DataInputStream dis =
	    new DataInputStream(new ByteArrayInputStream(buffer));
	 return parseMessage(dis);
      }
      catch (IOException e) {
	 Debug.showError("Some I/O Error", e);
	 throw new StorageException(StorageException.IOError);
      }    
   }

   /** Parse a message contained in a DatagramPacket. 
    * 
    * @exception StorageException Some error arose in parse the message
    */
   public static Message parseMessage(DatagramPacket p) 
      throws StorageException {
      return parseMessage(p.getData());
   }

   /** Transmit a packet to a DataOutputStream 
    *
    * @exception IOException There was a problem sending the raw data to the
    * DataOutputStream.  */
   public abstract void sendMessage(DataOutputStream dos)
      throws IOException;

   /** Create a byte array corresponding to the message 
    *
    * @exception IOException A problem arose creating a DataOutputStream.
    *             This should never happen.
    */
   public byte[] toByteArray() 
      throws IOException {
      ByteArrayOutputStream baob = new ByteArrayOutputStream();
    
      sendMessage(new DataOutputStream(baob));
      return baob.toByteArray();
   }
}

