/*
 * Copyright (c) 1998 by Interdisciplinary Center Herzliya
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE INTERDISCIPLINARY CENTER HERZLIYA BE LIABLE
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND
 * ITS DOCUMENTATION, EVEN IF THE INTERDISCIPLINARY CENTER HERZLIYA
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE INTERDISCIPLINARY CENTER HERZLIYA SPECIFICALLY DISCLAIMS ANY
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE 
 * SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE
 * INTERDISCIPLINARY CENTER HERZLIYA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 * */

// $Header: /usr/u/raoul/cvs/cs415-storage-server/il/ac/idc/storage/StorageException.java,v 1.1.1.1 1999/09/17 18:07:27 raoul Exp $ 

package il.ac.idc.storage;

import java.io.*;

public class StorageException extends RuntimeException {
  private int errCode;

  /** Some sort of IOError occured when processing this
      operation */
  static public final int IOError         = -1;
  /** The packet received could not be parsed */
  static public final int BadWireAPI      = -2;
  /** It was not possible to create a Data Storage Unit */
  static public final int CreateDataFail  = -3;
  /** It was not possible to create a Directory Storage Unit */
  static public final int CreateDirFail   = -4;
  /** An attempt to delete a storage unit failed for some reason */
  static public final int DeleteFail      = -5;
  /** It was not possible to read the desired data storage.  
    * Perhaps the data storage unit didn't exist. */
  static public final int ReadDataFail    = -6;
  /** Blocks badly allocated (overlapping) */
  static public final int BadBlockAlloc   = -7;
  /** Wrong type of Storage Unit while doing restore */
  static public final int UnitMismatch    = -8;
  /** Expected a StorageID, but didn't receive one */
  static public final int NotStorageID    = -9; 
  /** The Index file was not found */
  static public final int IndexFNF        = -10;
  /** Deleting a Storage Unit that is not registered */
  static public final int DeleteNExistSU  = -11;
  /** Read of uninitialized portion of a Data Storage unit */
  static public final int ReadUninitialized = -12;
  /** Attempt to bind a name in a directory failed */
  static public final int BindFailed = -13;
  /** Attempt to unbind a name in a directory failed */
  static public final int UnBindFailed = -14;
  /** Attempt to send a message that is incompatible with the 
      underlying network transfer protocol */
  static public final int MessageNetworkIncompatible = -15;

  static public final int Unimplemented   = -255;

  public StorageException(int i) {
    errCode = i;
  }

  /** Signal an I/O error as a StorageException. Print the I/O error
    * to the error stream, and then signal a StorageException.
    *
    * @exception StorageException Always thrown
    */
  public static void signalIOError(Exception e) 
    throws StorageException {
    Debug.showError("Some I/O error", e);
    throw new StorageException(StorageException.IOError);
  }

  
  public int getErrorCode() {
    return errCode;
  }

  /** Returns a string that describes the error condition
    */
  public String getMessage() {
    switch(errCode) {
    case IOError:
      return "Storage Sever Error: Some I/O Error";

    case BadWireAPI:
      return "Storage Server Error: Illegal Wire API message format";

    case CreateDataFail:
      return "Storage Server Error: Create Data Storage Unit failed";

    case CreateDirFail:
      return "Storage Server Error: Create Directory Storage Unit failed";

    case DeleteFail:
      return "Storage Server Error: Delete Storage Unit failed";

    case ReadDataFail:
      return "Storage Server Error: Reading from a Data Store failed";

    case BadBlockAlloc:
      return "Storage Server Error: Blocks badly allocated (overlapping)";

    case UnitMismatch:
      return "Storage Server Error: Wrong type of Storage Unit while doing restore";

    case NotStorageID:
      return "Storage Server Error: Expected a StorageID and didn't get it";

    case IndexFNF:
      return "Storage Server Error: Missing Index file";

    case Unimplemented:
      return "Storage Server Error: Unimplemented branch of system encountered";

    case DeleteNExistSU:
      return "Deleting a Storage Unit that is not registered";

    case ReadUninitialized:
      return "Read of uninitialized portion of a Data Storage unit";

    case BindFailed:
      return "Cannot bind the directory and the StorageUnit for some reason";

    case UnBindFailed:
      return "Cannot unbind the directory and the StorageUnit for some reason";

    case MessageNetworkIncompatible:
      return "Attempt to send a message that is incompatible with the underlying network transfer protocol";

    default:
      return "Unknown type of Storage Exception: " + errCode;
    }
  }
}
