package disksim;

import java.io.*;
import disksim.DiskModel;

class Statistics {
    int nr_reads = 0;
    int nr_writes = 0;
    int nr_blocks_read = 0;
    int nr_blocks_written = 0;
    int read_time = 0;
    int write_time = 0;
    int total_time = 0;

    void reset() {
	nr_reads = 0;
	nr_writes = 0;
	nr_blocks_read = 0;
	nr_blocks_written = 0;
	read_time = 0;
	write_time = 0;
	total_time = 0;
    }
}

class DiskLogger {
    private int mode;
    private PrintWriter thelog = null;

    DiskLogger(int mode, String logfile) {
	switch (mode) {
	case BlockingDiskModel.MODE_SILENT:
	case BlockingDiskModel.MODE_DEBUG:
	case BlockingDiskModel.MODE_STATS:
	    this.mode = mode;
	    break;
	default:
	    System.err.println("DiskLogger: illegal mode " + mode);
	    System.exit(1);
	}

	if (mode != BlockingDiskModel.MODE_SILENT) {
	    System.out.println("HEY");
	    if (logfile == null) {
		logfile = "disksim.log";
	    }
	    try {
		thelog = new PrintWriter(new FileWriter(logfile, true));
	    } catch (IOException e) {
		System.err.println("DiskLogger: cannot open logfile " +
				   logfile + ": " + e.getMessage());
		System.exit(1);
	    }
	    log("=== disk simulator log ===");
	}
    }

    void log(String str) {
	System.out.println("MODE == " + mode);
	if (mode == BlockingDiskModel.MODE_DEBUG) {
	    thelog.println(str);
	}
    }

    void statistics(Statistics stats) {
	System.out.println("STATS");
	if (mode != BlockingDiskModel.MODE_SILENT) {
	    thelog.println("=== statistics ===");
	    thelog.println("reads " + stats.nr_reads);
	    thelog.println("writes " + stats.nr_writes);
	    thelog.println("nr_blocks_read " + stats.nr_blocks_read);
	    thelog.println("nr_blocks_written " + stats.nr_blocks_written);
	    thelog.println("read time " + stats.read_time);
	    thelog.println("write time " + stats.write_time);
	    thelog.println("total time " + stats.total_time);
	    thelog.flush();
	}
    }
}

public class BlockingDiskModel {
    public static final int SECTOR_SIZE = DiskModel.SECTOR_SIZE;
    public static final int MODE_SILENT = 0;
    public static final int MODE_DEBUG = 1;
    public static final int MODE_STATS = 2;

    private DiskLogger logger;
    private Statistics stats;
    private DiskModel dm;

    public BlockingDiskModel(int mode, String logfile) {
	logger = new DiskLogger(mode, logfile);
	stats = new Statistics();
	stats.reset();
	dm = new DiskModel();
	dm.disk_init();
    }

    public void read(long offset, int quant) {
	/** blocks the amount of time that it would take to read the given
	 * number of sectors at the given address.
	 *
	 *@param offset the sector to start reading at
	 *@param quant the number of sectors to read */
	Thread t = Thread.currentThread();
	long us = dm.faux_read(offset,quant);
	try{
	    t.sleep(us / 1000, (int)(us % 1000) * 1000 ); 
	    // sleep works in units of milliseconds and nanoseconds, not
	    // microseconds
	} catch (InterruptedException e) {}

	logger.log("read " + offset + " " + quant + " " + us);
	stats.nr_reads++;
	stats.nr_blocks_read += quant;
	stats.read_time += us;
	stats.total_time += us;
    }

    public void write(long offset, int quant, boolean blocking) {
	/** blocks the amount of time that it would take to write the given
	 * number of sectors at the given address.
	 *
	 * @param offset the sector to start writing at
	 * @param quant the number of sectors to write
	 * @param blocking whether or not to do a blocking (i.e., synchronous)
	 *        write.  
	 */
	Thread t = Thread.currentThread();
	long us = dm.faux_write(offset,quant, (blocking ? 1 : 0));
	try{
	    t.sleep(us / 1000, (int)(us % 1000) * 1000 ); 
	    // sleep works in units of milliseconds and nanoseconds, 
	    // not microseconds
	} catch (InterruptedException e) {}

	logger.log("write " + offset + " " + quant + " " + us);
	stats.nr_writes++;
	stats.nr_blocks_written += quant;
	stats.write_time += us;
	stats.total_time += us;
    }

    public void resetStatistics() {
	stats.reset();
	logger.log("reset statistics");
    }

    public void printStatistics() {
	logger.statistics(stats);
    }
}
