/*
 * Copyright (c) 1998 by Interdisciplinary Center Herzliya
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE INTERDISCIPLINARY CENTER HERZLIYA BE LIABLE
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND
 * ITS DOCUMENTATION, EVEN IF THE INTERDISCIPLINARY CENTER HERZLIYA
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE INTERDISCIPLINARY CENTER HERZLIYA SPECIFICALLY DISCLAIMS ANY
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE 
 * SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE
 * INTERDISCIPLINARY CENTER HERZLIYA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 * */

// $Header: /usr/u/raoul/cvs/cs415-storage-server/il/ac/idc/storage/server/DataStore.java,v 1.3 1999/09/28 20:16:38 raoul Exp $ 

package il.ac.idc.storage.server;

import il.ac.idc.storage.*;
import java.io.*;
import java.util.*;
//
//
// DataStore
//
//
public class DataStore extends StorageUnit {
    int freeByte;		      // First free byte in data file
    boolean blocksWritten;      // Whether or not the index file is up to date
    Vector blockRecords;
  
    File dataFN;	      // File Name for data store

    DataStore(SimpleStorageServer s, StorageID storID) {
	server = s;
	sid = storID;
	indexFN = new File(server.directory, 
			   storID.toLong() + ".index");
	dataFN = new File(server.directory, 
			  storID.toLong() + ".data");
	blockRecords = new Vector(3);
	freeByte = 0;
    }
  
    /** Reconstruct a Storage Unit from the data stored on Disk 
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public void restoreStorageUnit() 
	throws StorageException {
	DataInputStream file;
	try {
	    file = new DataInputStream(new FileInputStream(indexFN));
      
	    if (StorageUnit.DataStore != file.readByte()) {
		throw new StorageException(StorageException.UnitMismatch);
	    }

	    try {
		for(;;) {
		    BlockRecord record = new BlockRecord(file);
		    blockRecords.addElement(record);
		    freeByte = Math.max(freeByte, record.physicalStart + record.length);
		    blocksWritten = true;
		}
	    }
	    catch (EOFException e) { // Reach end of file.  
		file.close();
	    }
	}
	catch (IOException e) {
	    StorageException.signalIOError(e);
	}
    }
  
    /** Saves all of the in-memory information about the storage unit to disk so
	that it can be restored by a restoreStorageUnit command. 
	*
	* @exception StorageException CatchAll for errors in the StorageSystem
	*/
    public void saveStorageUnit() 
	throws StorageException {
	if (blocksWritten != true) {
	    try {
		// The data portion of the storage unit is assumed to be already 
		// written to disk. Only the index information is actually saved to 
		// disk at this point.
	
		int numBlocks = blockRecords.size();
		DataOutputStream file 
		    = new DataOutputStream(new FileOutputStream(indexFN));
		// The first byte written indicates whether this is a data storage unit
		// or a directory storage unit.
		file.writeByte(DataStore);    // This is a data storage unit      
		for (int i = 0; i < numBlocks; i++) {
		    ((BlockRecord)blockRecords.elementAt(i)).writeBlock(file);
		}
	    }
	    catch (IOException e) {
		StorageException.signalIOError(e);
	    }
	    blocksWritten = true;
	}
    }

    /** Deletes a backing store for the storage unit from the disk.
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public void deleteUnit() 
	throws StorageException {
	dataFN.delete();		// Delete the main storage
	super.deleteUnit();	// Create a canonical DeadFile
    }

    /** Allocate a new BlockRecord and creates and initializes the necessary 
     * storage on the server.
     *
     * @param virtualAddress Starting address for the block
     * @param size The length of the block 
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public boolean newBlockRecord(long virtualAddress, int size) 
	throws StorageException {
	BlockRecord br = new BlockRecord(virtualAddress, freeByte, size);
    
	// Successfully created the storage, now update the index file
	for (int i = 0; i < blockRecords.size(); i++) {
	    BlockRecord obr = (BlockRecord)blockRecords.elementAt(i);
	    if (obr.virtualStart + obr.length <= br.virtualStart) {
		// Do nothing. New block follows this block
	    } else if (br.virtualStart + br.length <= obr.virtualStart) {
		// New block precedes this block.  Insert it
		blockRecords.insertElementAt(br, i);
		br = null;
		break;
	    } else {
		// The blocks overlap.  This is an error.
		throw new StorageException(StorageException.BadBlockAlloc);
	    }
	}
	if (br != null) {
	    blockRecords.addElement(br);
	}
	freeByte = freeByte + size;
	blocksWritten = false;
    
	return true;
    }

    private BlockRecord findBlockRecord(long vad) {
	for (int i = 0; i < blockRecords.size(); i++) {
	    BlockRecord br = (BlockRecord)blockRecords.elementAt(i);

	    if (br.virtualStart <= vad && vad < br.virtualStart + br.length) {
		return br;
	    }
	}
	return null;
    }

    private BlockRecord findOverlappingBlock(long vad, long size) {
	for (int i = 0; i < blockRecords.size(); i++) {
	    BlockRecord br = (BlockRecord)blockRecords.elementAt(i);

	    if (vad <= br.virtualStart && br.virtualStart < vad + size) {
		return br;
	    }
	}
	return null;
    }


    /** Read some bytes from the disk.  Reads no more than size bytes.
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
    public byte[] read(long vad, long size) 
	throws StorageException {
	BlockRecord br = findBlockRecord(vad);

	if (br == null) 
	    throw new StorageException(StorageException.ReadUninitialized);

	// The following is the offset into the 
	int blockOffset = (int)(vad - br.virtualStart);
	int nBytes = (int)Math.min(br.length - blockOffset, size);
    
	try {
	    RandomAccessFile dfile = new RandomAccessFile(dataFN, "r");
      
	    byte[] buffer = new byte[nBytes];
	    dfile.seek(br.physicalStart + blockOffset);
	    dfile.read(buffer, 0, nBytes);
	    dfile.close();
	    return buffer;
	}
	catch (IOException e) {
	    Debug.println(0,"bytes read: " + nBytes 
			  + " physicalStart: " + br.physicalStart
			  + " blockOffset: " + blockOffset);
	    Debug.showError(0,"Some I/O Exception while reading a DataStore", e);
	    return null;
	}
    }

    /** Write size bytes to the disk, starting at offset from 
     * the beginning of buffer. This may need to allocate addition 
     * blocks.  
     *
     * @exception StorageException CatchAll for errors in the StorageSystem
     */
public boolean write(long vad, byte[] buffer, int start, int size)
	throws StorageException {
	long write_size;
	long write_pos;
	int block_index = 0;
	BlockRecord br = null;

	try {
	    RandomAccessFile dfile = new RandomAccessFile(dataFN, "rw");

	    while (size > 0) {
		// find a block that overlaps with the next segment to
		// write or that follows the segment.
		for ( ; block_index < blockRecords.size(); block_index++) {
		    br = (BlockRecord) blockRecords.elementAt(block_index);
		    if (vad < br.virtualStart + br.length) {
			break;
		    }
		}
		
		if (block_index < blockRecords.size()) {
		    // we found a block; br is valid
		    if (vad < br.virtualStart) {
			// insert new block before current block
			write_size = Math.min(size, br.virtualStart - vad);
			write_pos = (long) freeByte;
			newBlockRecord(vad, (int) write_size);
			block_index++;
		    } else if (vad < br.virtualStart + br.length) {
			// overwrite end of current block
			write_size = Math.min(size,
					      br.virtualStart + br.length - vad);
			write_pos = br.physicalStart + vad - br.virtualStart;
		    } else {
			throw new StorageException(StorageException.BadBlockAlloc);
		    }
		} else {
		    // we've exhausted all existing blocks
		    write_size = size;
		    write_pos = (long) freeByte;
		    newBlockRecord(vad, (int) write_size);
		}
		
		dfile.seek(write_pos);
		dfile.write(buffer, start, (int) write_size);

		size -= write_size;
		vad += write_size;
		start += write_size;
	    }
	    dfile.close();
	} catch (IOException e) {
	    return false;
	}
	return true;
    }
}

