/*
 * Copyright (c) 1998 by Interdisciplinary Center Herzliya
 * All rights reserved.
 *
 * Permission is hereby granted, without written agreement and without
 * license or royalty fees, to use, copy, modify, and distribute this
 * software and its documentation for any purpose, provided that the
 * above copyright notice and the following two paragraphs appear in
 * all copies of this software.
 *
 * IN NO EVENT SHALL THE INTERDISCIPLINARY CENTER HERZLIYA BE LIABLE
 * TO ANY PARTY FOR DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR
 * CONSEQUENTIAL DAMAGES ARISING OUT OF THE USE OF THIS SOFTWARE AND
 * ITS DOCUMENTATION, EVEN IF THE INTERDISCIPLINARY CENTER HERZLIYA
 * HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * THE INTERDISCIPLINARY CENTER HERZLIYA SPECIFICALLY DISCLAIMS ANY
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES 
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE 
 * SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE
 * INTERDISCIPLINARY CENTER HERZLIYA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 * */

// $Header: /usr/u/raoul/cvs/cs415-storage-server/il/ac/idc/storage/client/UDPStorageServer.java,v 1.1.1.1 1999/09/17 18:07:30 raoul Exp $ 

package il.ac.idc.storage.client;

import il.ac.idc.storage.*;
import java.net.*;
import java.io.*;

/** Implements the StorageServer interface, as a client, communicating
  * through a UDP channel.
  */
abstract public class UDPStorageServer implements StorageServer {
  private InetAddress server;
  private DatagramSocket socket;

  /** Create a reference to a StorageServer, and cache a copy of the
    * user's credentials 
    *
    * @param address The internet address of the server.
    * @exception IOException Some problem arose dealing with the underlying 
    *             file system.
    */
  public UDPStorageServer(InetAddress address) 
    throws IOException {
     // Must validate address, by making sure that the address refers
     // to a valid Shared Storage Server
     server = address;
     socket = new DatagramSocket();
   }

  /** Transmit a Message on the server's socket and parse the result.
    *
    * @exception StorageException CatchAll for errors in the StorageSystem
    */
  private Message transmit(Message out) 
    throws StorageException {
    Message mess;
    try {
      byte[] data = out.toByteArray();
      byte[] buffer = new byte[Message.UDPMessageSize];

      Debug.println("Sending: " + out);      

      // If the message size is larger than the packet size, throw an exception.
      if (data.length >= Message.UDPMessageSize) {
        throw new StorageException(StorageException.MessageNetworkIncompatible);
      }

      DatagramPacket dgp
        = new DatagramPacket(data, data.length, server, Message.UDPPort);
      DatagramPacket packet = new DatagramPacket(buffer, buffer.length);
       
      socket.send(dgp);       
      socket.receive(packet);

      mess = Message.parseMessage(packet);
      Debug.println("  Received: " + mess);
      return mess;
    }
    catch(IOException e) {
      StorageException.signalIOError(e);
      return null;
    }
  }
     
  /** Create a <EM>data</EM> storage unit on this Storage Server that can
    * contain at least size number of bytes of data.  
    * 
    * @param size the number of bytes in the storage unit that is created.
    * @return the StorageID of the newly created data storage unit
    * @exception StorageException CatchAll for errors in the StorageSystem
    */
  public StorageID createDataStore(Credentials cred, long size) 
   throws StorageException {   
     Message response = transmit(new Create(cred, StorageID.none, size));

     if (response.marker == Message.Create) {
       return ((Create)response).id;
     } else throw new StorageException(StorageException.CreateDataFail);
   }

  /** Create a <EM>directory</EM> storage unit on this Storage Server that can
    * contain at least size number of bytes of data.  
    * 
    * @param size the number of bytes in the storage unit that is created.
    * @return the StorageID of the newly created data storage unit
    * @exception StorageException CatchAll for errors in the StorageSystem
    */
  public StorageID createDirectoryStore(Credentials cred) 
     throws StorageException {
      Message response 
        = transmit(new CreateDirectory(cred, StorageID.none));

      if (response.marker == Message.CreateDir) {
        return ((CreateDirectory)response).id;
      } else throw new StorageException(StorageException.CreateDirFail);
  }

  /** Delete a storage unit. 
    *
    * @return an acknowledgement code.
    * @exception StorageException CatchAll for errors in the StorageSystem
    */
  public void deleteUnit(Credentials cred, StorageID id) 
    throws StorageException {
       Message response = transmit(new Delete(cred, id));

       expectAcknowledge(response);
    }

  /** Reads the data contained in the storage unit and returns it as
    * an array of bytes. The number of bytes read may be less than
    * len. If an error occurs, an exception is thrown. 
    *
    * @exception StorageException CatchAll for errors in the StorageSystem
    */
  public byte[] readData(Credentials cred, StorageID unitID, long start, long len) 
    throws StorageException {
    Message response = transmit(new ReadData(cred, unitID, start, len));
    
    if (response.marker == Message.WriteData) {
      return ((WriteData)response).data;
    } else {
      expectAcknowledge(response);
      return null;      // Should always thrown in previous statement
    }
  }

  /** Writes the indicated data into the storage unit at the indicated
    * point.  The first len bytes in data are written. The value
    * returned is the return value of the acknowledgement. 
    *
    * @exception StorageException CatchAll for errors in the StorageSystem
    */
  public void writeData(Credentials cred, StorageID unitID, long start, long len, byte[] data) 
    throws StorageException {
      Message response 
        = transmit(new WriteData(cred, unitID, start, len, data));

      expectAcknowledge(response);
  }

  /** Checks to see if the message is an an acknowledement message. 
    * If not, or if the acknowledgement code is not zero, signals 
    * StorageException.
    *
    * @exception StorageException Signaled whenever the message is not 
    *                             a positive acknowledgement
    */
  private void expectAcknowledge(Message mess) 
    throws StorageException {
    if (mess.marker != Message.Acknowledge) {
      System.err.println("Wrong Class message received: " + mess);
      throw new StorageException(StorageException.IOError);
    } else {
      Acknowledge ack = (Acknowledge)mess;

      if (ack.data != 0)
        throw new StorageException(ack.data);
    }
  }
 
  /** Read an attribute of the storage unit 
    *
    * @exception StorageException CatchAll for errors in the StorageSystem
    */
  public byte[] readAttribute(Credentials cred, StorageID unitID, int Attribute)
    throws StorageException {
    return null;
  }

  /** Writes an attribute of the storage unit. The return value is the
    * return value of the acknowledgement. 
    *
    * @exception StorageException CatchAll for errors in the StorageSystem
    */
  public int writeAttribute(Credentials cred, StorageID unitID, int Attribute, byte[] data)
    throws StorageException {
    return -1;
  }
        
  /** Lookups up a string in a directory and returns the StorageID of 
    * the corresponding StorageUnit. 
    *
    * @param cred The Credentials of the originator of the query
    * @param directory The StorageID of the directory to be searched
    * @param unitName The String to be looked up in the directory
    * @exception StorageException CatchAll for errors in the StorageSystem
    */

  public StorageID lookup(Credentials c, StorageID directory, String unitName)
    throws StorageException {
          Message response = transmit(new Lookup(c, directory , unitName));

          if (response.marker == Message.Bind) {
            StorageID newID = ((Bind)response).id;

            if (newID.toLong() == 0) return null;
            else return newID;
          } else if (response.marker == Message.Acknowledge) {
            Acknowledge ack = (Acknowledge)response;

            throw new StorageException(ack.data);
          } else {
            throw new StorageException(-250);
          }
  }

  /** Binds the storage unit to a string name in the directory storage
    * unit.  
    *
    * @param cred The Credentials of the originator of the query
    * @param directory The StorageID of the directory to be searched
    * @param unitName The String to be looked up in the directory
    * @exception StorageException CatchAll for errors in the StorageSystem
    */

  public void bind(Credentials cred, StorageID directory, String unitName, 
                   StorageID unit)
    throws StorageException {
    Message response = transmit(new Bind(cred, directory , unit , unitName));

    if (response.marker != Message.Acknowledge) 
      throw new StorageException(StorageException.BindFailed);
  }
                
  /** Unbinds a string name in the directory storage unit. 
    *
    * @param cred The Credentials of the originator of the query
    * @param directory The StorageID of the directory to be searched
    * @param unitName The String to be looked up in the directory
    * @exception StorageException CatchAll for errors in the StorageSystem
    */

  public void unbind(Credentials c, StorageID directory, String unitName)
    throws StorageException {
    Message response = transmit(new Unbind(c, directory , unitName));

    if (response.marker != Message.Acknowledge) 
      throw new StorageException(StorageException.UnBindFailed);
  }

  /** Locks the region of the storage unit for exclusive use by the
    * current client. Any negative acknowledgement is indicated by 
    * throwing a StorageException
    *
    * @param cred The Credentials of the originator of the query
    * @param unit The StorageID of the Storage Unit to be locked
    * @param start The first byte to be locked 
    * @param length The number of bytes to be locked
    * @exception StorageException Indicates a negative acknowledgement
    */
  public void lock(Credentials c, StorageID unit, long start, long length) 
    throws StorageException {
  }
                
  /** Unlocks the region of the storage unit for exclusive use by the
    * current client. Any negative acknowledgement is indicated by 
    * throwing a StorageException
    *
    * @param cred The Credentials of the originator of the query
    * @param unit The StorageID of the Storage Unit to be unlocked
    * @param start The first byte to be locked 
    * @param length The number of bytes to be unlocked
    * @exception StorageException Indicates a negative acknowledgement
    */
  public void unlock(Credentials c, StorageID unit, long start, long length) 
    throws StorageException {
  }

}

