/**********
 * 
 * diskmodel.c - 
 *     Model the behavior of an HP 97560 hard drive
 * 
 * Public interface:
 *   disk_init()
 *   faux_read()
 *   faux_write()
 * 
 * Nils Nieuwejaar 1995-96
 *
 *********/

/* Things to look for:
            WAS A TEST
            DUBIOUS
*/
package disksim;

import java.io.*;


// these are basically structs
class Timeval {
  public long tv_sec,tv_usec;

  public Timeval () {
    tv_sec = 0;
    tv_usec = 0;
  }
  
  // Computes the difference in time between a and b in microseconds
  static long diff(Timeval a, Timeval b) {
    return ((b.tv_sec - a.tv_sec) * 1000000 + (b.tv_usec - a.tv_usec));
  }
  
  static long cmp(Timeval a, Timeval b) {
    return ((a.tv_sec < b.tv_sec) ? -1 :
            ((a.tv_sec > b.tv_sec) ? 1 : (a.tv_usec - b.tv_usec)));
  }

  public void add_time(Timeval currentTime, long us)
  {
    this.tv_usec = currentTime.tv_usec + us;
    this.tv_sec = currentTime.tv_sec;
    
    while (this.tv_usec > 1000000) {
      this.tv_usec -= 1000000;
      this.tv_sec++;
    }
  }

  // Set the time to the current time.
  public void getTimeOfDay() {
    long msec = System.currentTimeMillis();

    this.tv_sec = msec / 1000;
    this.tv_usec = 1000*(msec % 1000);
  }
}

class DiskRegion {
  public long lstart, lend;
  public long pstart, pend;
  public long length;

  public DiskRegion() {
    lstart = 0;
    lend = 0;
    pstart = 0;
    pend = 0;
    length = 0;
  }
}

class DiskAddress implements Cloneable {
  public long cylinder;
  public long track;
  public long head;
  public long logical, physical;

  public Object clone() {
    try {
      return super.clone();
    } catch (CloneNotSupportedException e) {
      return null;
    }
  }
}

class DiskState {
  public DiskAddress add;
  public Timeval last_time;
  public Timeval ready_time;

  public DiskState() {
    add = new DiskAddress();
    last_time = new Timeval();
    ready_time = new Timeval();
  }
}

class CacheState {
  public int valid;
  public long last;
  public DiskAddress first;
  public DiskAddress prefetch;
  public Timeval start;
  public Timeval done;

  public CacheState() {
    valid=0;
    last=0;
    first = new DiskAddress();
    prefetch = new DiskAddress();
    start = new Timeval();
    done = new Timeval();
  }
}


public class DiskModel {
  /* All parameters are from ruemmler:model2 and hp:97560 */

  static final int SECTOR_SIZE = 512;   /* bytes */
  static final int SEC_PER_TRACK = 72;
  static final int TRACKS_PER_CYL = 19;
  static final int NUM_REGIONS = 3;

  /* cylinder skew of the disk in sectors */
  static final int CYLINDERSKEW = 18;
  /* track skew of the disk in sectors */
  static final int TRACKSKEW = 8;

  /* Combine bus and controller overhead into a single number */
  static final int OVERHEAD = 2300;  /* microseconds */

  static final int HEAD_SWITCH = 2500;  /* microseconds */
  static final int CYLINDER_SWITCH = 2500;  /* microseconds */

  /* 10 MB/sec -> bytes per us */
  static final double BUSSPEED = (10.0 * 1024.0 * 1024.0 / 1000000.0) ;

  static final int DISKRPM = 4002;
  static final double USEC_PER_REV = (60.0 * 1000000.0 / DISKRPM);
  static final double USEC_PER_SECTOR = USEC_PER_REV / SEC_PER_TRACK;

  /* write fence size */
  static final double WRITEFENCE = 64 * 1024.0; /* bytes */
  static final double write_fence_time = WRITEFENCE / BUSSPEED;  
  /* usec */

  /* size of disk's on-board cache in KB */
  static final int CACHE_SIZE = 128;

  /* size of disk's on-board cache in sectors */
  static final int CACHE_SLOTS = (CACHE_SIZE * 1024 / SECTOR_SIZE);


  DiskRegion[] regions = new DiskRegion[NUM_REGIONS];
  DiskState ds;
  CacheState cs;
  Timeval now;

  public DiskModel() {
    int i;
    for (i=0; i<regions.length ;i++) {
      regions[i] = new DiskRegion();
    }
    ds = new DiskState();
    cs = new CacheState();
    now = new Timeval();
    disk_init();
  }

  
  /*
   * Initialize the state of the disk.
   * Set up the logical.physical region mappings
   */
  public void disk_init()
  {
    int i;
    long t;

    regions[0].pstart = (TRACKS_PER_CYL + 4) * SEC_PER_TRACK;
    regions[0].pend = (646*TRACKS_PER_CYL + 4) * SEC_PER_TRACK - 1;
    
    regions[1].pstart = 654*TRACKS_PER_CYL * SEC_PER_TRACK;
    regions[1].pend = (1298*TRACKS_PER_CYL + 19) * SEC_PER_TRACK - 1;
    
    regions[2].pstart = 1308*TRACKS_PER_CYL * SEC_PER_TRACK;
    regions[2].pend = (1952*TRACKS_PER_CYL + 19) * SEC_PER_TRACK - 1;

    for (t=0, i=0; i<3; i++) {
      regions[i].length = regions[i].pend - regions[i].pstart + 1;
      regions[i].lstart = t;
      t += regions[i].length;
      regions[i].lend = t-1;
    }
    ds.ready_time.tv_usec = 0;
    ds.ready_time.tv_sec = 0;
    ds.last_time.tv_usec = 0;
    ds.last_time.tv_sec = 0;
    ds.add.cylinder = 0;
    ds.add.track = 0;
    ds.add.logical = 0;
    ds.add.physical = 0;
    cs.valid = 0;
    
    ds.last_time.getTimeOfDay();
  }


  /*
   * How long will it take to move the head mechanism to the proper
   * cylinder and head?
   * 
   * 1) If the mechanism is already in place, we only have to change heads
   * 2) If we have to move a short distance, most of the time is spent
   *    accelerating.  Thus the time is proportional to the sqrt of the
   *    distance.
   * 3) Long seeks spend most of the time moving at a constant rate, after
   *    a (basically) constant time accellerating.
   */

  long seek_time(DiskAddress add) {
    long cyl, tracks, us;
    
    cyl = (long) Math.abs(add.cylinder - ds.add.cylinder);
    tracks = Math.abs(ds.add.head - add.head);

    if (cyl == 0) {
      if (tracks != 0)
        us = 2500;
      else
        us = 0;
    }
    else if (cyl < 383)
      us = (long)(3240 + 400 * Math.sqrt((double)cyl));
    else
      us = (long)(8000 + 8 * cyl);
    // DUBIOUS -- casts to long
    
    //  debug(("Seektime.  cyl: %u  tracks: %u  us: %u\n", cyl, tracks, us));
    return(us);
  }



  /* Once the head is positioned, how long does it take for the proper
   * sector to rotate under the head?  */


  long rot_delay(DiskAddress add, Timeval start, long seektime) {
    double revs, dest, rot;
    long us, past, ss;

    past = Timeval.diff(ds.last_time, start);

    /* Total revolutions since the last time we checked */
    revs = (double) (Timeval.diff(ds.last_time, start) + seektime) 
      / USEC_PER_REV;

    /* Disregard whole revolutions - we're back where we started */
    /* This might be clearer if an explicit truncation were used instead of a cast. */
    revs = revs - (long) revs;

    ss = add.physical % SEC_PER_TRACK;
    dest = (double) (ss + add.cylinder * CYLINDERSKEW +
                     (add.track - add.cylinder) * TRACKSKEW) /
      SEC_PER_TRACK;
    
    /* How far is it from where we are, to where we're going? */
    rot = 1 + dest - revs;
    rot = rot - (long) rot;
    
    us = (long)(seektime + (rot * USEC_PER_REV));
    //  debug(("rot_delay: us: %u\n", us-seektime));
    return(us-seektime);
  }

  /* How long before the head is right over the data we want? */
  long move_time(DiskAddress add, Timeval start) {
    long us, s, r;

    s = seek_time(add);
    r = rot_delay(add, start, s);

    //  debug(("   Seek: %u\tRot: %u\n", s, r));
    return(s+r);
  }

  /*
 * Move the head to the new address, and update the time
 */
  void move_to(DiskAddress add, long us) {
    ds.add = (DiskAddress)add.clone();
    ds.last_time.add_time(now,us); // Note, ds.last_time is the target
  }

  /*
 * Once the head is in position, how long will it take to actually
 * transfer the data to disk?
 */
  long transfer_time(DiskAddress add, long sectors) {
    DiskAddress end = new DiskAddress();
    long cyl, track;

    translate(add.logical + sectors, end);

    cyl = end.cylinder - add.cylinder;
    track = end.track - add.track - cyl;

    /* DUBIOUS cast */
    return (long)(sectors * USEC_PER_SECTOR +
                  cyl * CYLINDER_SWITCH +
                  track * HEAD_SWITCH);
  }


  /*
   * Given a logical block number, translate it into a (cylinder,
   * head, sector) number
   */
  void translate(long logical, DiskAddress add) {
    int r;
    long phys;

    r = 0;
    while ((logical - regions[r].lend) > 0 && r < NUM_REGIONS)
      r++;

    //    debug(("%u %d %u\n", logical, r, (logical - regions[r].lstart)));
    phys = regions[r].pstart + (logical - regions[r].lstart);
    add.track = (long) (phys / (int) SEC_PER_TRACK);
    add.head = add.track % TRACKS_PER_CYL;
    add.cylinder = add.track / TRACKS_PER_CYL;
    add.logical = logical;
    add.physical = phys;
  }



  /*
 * Handle the disk's cache.  This gets pretty hairy.
 * Among the factors we need to handle: Is the data all the way
 * in the cache?  Are we currently loading the cache?  Are we loading
 * the data we want?
 * 
 * Basically, it boils down to 'when will the data be ready'?
 */
  long in_cache(DiskAddress add, int sectors) {
    long first, last, done;
    long working, needed, ttime;
    long have;

    first = add.logical;
    last = add.logical + sectors;

    working = Timeval.diff(cs.start, now);
    //  debug(("Start: %u Prefetch: %u Working: %u Want: %u\n",
    //     cs.first.logical, cs.prefetch.logical, working,
    //     add.logical));

    /* Is there any chance at all that this is in the cache?
     * If not, skip all this and start over */
    if (cs.valid == 0 || (first < cs.first.logical) ||
        (first > cs.first.logical+CACHE_SLOTS)) {
      return(-1);
    }

    /* Is the whole thing already in the cache? */
    if (Timeval.cmp(now, cs.done) >= 0 && last <= cs.first.logical+CACHE_SLOTS)
      return(0);

    /* How long has the disk been prefetching? */
    working = Timeval.diff(cs.start, now);
    
    /* How long will it take to satisfy this request? */
    if (last < cs.prefetch.logical)
      needed = 0;
    else {
      /* If prefetching hasn't reached the start of this req, start over */
      if (first > cs.prefetch.logical &&
          transfer_time(cs.prefetch, first - cs.prefetch.logical) > working)
        {
          //      debug(("Not here yet\n"));
          return(-1);
        }
      needed = transfer_time(cs.prefetch, last - cs.prefetch.logical);
    }

    /* If we aren't already prefetching, get it started */
    if (Timeval.cmp(now, cs.done) >= 0) {
      have = cs.first.logical + CACHE_SLOTS - add.logical;
      //if (have != 0)
      //      debug(("Had %d sectors\n", have));
      ttime = transfer_time(add, CACHE_SLOTS) - transfer_time(add, have);
      translate(add.logical + have, cs.prefetch);
      cs.start.add_time(now, 0);
      cs.done.add_time(cs.start, ttime);
    }
    else {
      ttime = transfer_time(cs.prefetch,
                            (first-cs.prefetch.logical)+CACHE_SLOTS);
      done = transfer_time(cs.prefetch, first - cs.prefetch.logical);
      cs.prefetch = (DiskAddress)add.clone();

      cs.start.add_time(cs.start, done);
      cs.done.add_time(cs.start, ttime-done);
      //    debug(("total: %u  sofar: %u\n", ttime, done));
    }

    cs.first = (DiskAddress)add.clone();

    if (needed < working) {
      //    debug(("Ready now\n"));
      return(0);
    }
    //  debug(("   needed: %u working: %u left: %d\n",
    //     needed, working, needed-working));
    return(needed - working);
  }    

  /*
 * Toss whatever is in the cache, and start loading our data.
 * Figure out when our data will be ready, and when the cache will be
 * fully loaded
 */
  void reset_cache(DiskAddress add, long wait) {
    cs.valid = 1;
    cs.first = (DiskAddress)add.clone();
    cs.prefetch = (DiskAddress)add.clone();

    
    cs.start.add_time(now, wait);
    cs.done.add_time(cs.start, transfer_time(add, CACHE_SLOTS));
  }

  /*
 * Given an address and a number of sectors, how many microseconds
 * will it take to move that data from the disk across the bus to
 * the memory?
 */
  long read_time(DiskAddress add, int sectors)
  {
    long disk_time;   /* Time to transfer all data to disk */
    long find_time;   /* Time to position head for start of read */
    long wait_time;   /* Time until disk is ready to be to */
    long sector_bus_time; /* Time to transfer one sector across the bus */
    long total, tmp;
    long cache_wait;

    sector_bus_time = (long)(SECTOR_SIZE / BUSSPEED);

    //  debug(("Read:\tSec: %5u\t",  add.logical));

    cache_wait = in_cache(add, sectors);

    /* If the data is in the cache, send it right across the bus */
    if (cache_wait == 0)
      return (long)((sectors * SECTOR_SIZE) / BUSSPEED);

    /* If the data is mostly in the cache, the bus-time will be the
     * dominant cost.  If only a small amount of the data is in the
     * cache, we'll spend most of our time waiting for the disk
     */
    if (cache_wait > 0)
      return (long)(Math.max(cache_wait + sector_bus_time,
                             (sectors * SECTOR_SIZE) / BUSSPEED));

    /* Nothing was in the cache.  We have to wait for any pending
     * writes to complete, then wait for the head to be positioned,
     * and then for our data to come off the disk.
     */
    disk_time = transfer_time(add, sectors);
    find_time = move_time(add, now);

    if (Timeval.cmp(now, ds.ready_time) >= 0)
      wait_time = 0;
    else
      wait_time = Timeval.diff(now, ds.ready_time);
    
    total = wait_time + find_time + disk_time + sector_bus_time;

    move_to(add, wait_time + find_time);

    reset_cache(add, wait_time + find_time);
    return(total);
  }

  /*
 * Given an address and a number of sectors, how many microseconds
 * will it take to move that data across the bus to either cache or
 * disk (depending on synchronous flag).
 */
  long write_time(DiskAddress add, int sectors, int sync) {
    long disk_time;   /* Time to transfer all data to disk */
    long find_time;   /* Time to position head for start of write */
    long wait_time;   /* Time until disk is ready to be written to */
    long bus_time;    /* Time spent transferring data across bus */
    long pre_time;    /* Time spent transferring data to cache before
                       * writing to disk */
    long done_time;   /* When will all data be on disk? */
    long return_time; /* When do we return to caller? */
    long tmp;
    int  bytes;

    bytes = sectors * SECTOR_SIZE;
    bus_time = (long)(bytes / BUSSPEED);
    disk_time = transfer_time(add, sectors);
    tmp = Timeval.cmp(now, ds.ready_time);

    /* If we get lucky, we can add this to the end of the data
     * already in the disk cache.
     */
    if (cs.valid == 2 && (cs.last+1) == add.logical && tmp<0) {
      /* WAS A TEST */
      cs.last = add.logical + sectors-1; 
      ds.ready_time.add_time(ds.ready_time, disk_time);
      return(bus_time);
    }
    /* WAS A TEST */
    cs.last = add.logical + sectors-1; 
    cs.valid = 2;

    /* Do we have to wait for the cache to be flushed? */
    if (tmp >= 0) {
      wait_time = 0;
      /* Cache is fine, start seeking */
      find_time = move_time(add, now);
    }
    else {
      wait_time = Timeval.diff(now, ds.ready_time);
      /* Can't seek until the cache is flushed */
      find_time = move_time(add, ds.ready_time);
    }

    //  debug(("   Now: (%u, %u)  Ready: (%u, %u)  Diff: %u\n",
    //     now.tv_sec, now.tv_usec,
    //     ds.ready_time.tv_sec, ds.ready_time.tv_usec, wait_time));

    /* We don't start writing immediately.  We wait until WRITEFENCE
     * bytes are ready for the disk.
     */
    if (bytes > (int) WRITEFENCE)
      pre_time = Math.max(Math.max(bus_time,find_time), (int) write_fence_time);
    else
      pre_time = Math.max(bus_time,find_time);

    /* When will everything be on disk?  After the cache is flushed,
     * the data arrives across the bus, the disk head seeks, and
     * we actually write to disk.  Whew.
     */
    done_time = wait_time + pre_time + disk_time;

    //  debug(("   wait: %u  pre: %u  find: %u disk: %u done: %u\n",
    //     wait_time, pre_time, find_time, disk_time, done_time));

    if (sync != 0)
      return_time = done_time;
    else
      return_time = wait_time + bus_time;

    ds.ready_time.add_time(now, done_time);
    move_to(add, wait_time + find_time);

    return(return_time);
  }

  public long faux_read(long logical, int sectors)
  {
    DiskAddress add = new DiskAddress();
    long usec;

    translate(logical, add);
    now.getTimeOfDay();

    /* encompasses controller overhead, grabbing the bus, etc */
    now.add_time(now, OVERHEAD);

    usec = OVERHEAD + read_time(add, sectors);
    return(usec);
  }

  public long faux_write(long logical, int sectors, int immediate) {
    DiskAddress add = new DiskAddress();
    long usec;

    translate(logical, add);
    now.getTimeOfDay();

    /* encompasses controller overhead, grabbing the bus, etc. */
    
    now.add_time(now, OVERHEAD);

    usec = write_time(add, sectors, immediate);
    return(usec);
  }


}


