#ifndef __MINISOCKETS_H_
#define __MINISOCKETS_H_

/*
 *	Definitions for minisockets.
 *
 *      You should implement the functions defined in this file, using
 *      the names for types and functions defined here. Functions must take
 *      the exact arguments in the prototypes.
 *
 *      Put the initialization code in minimsg_initialize together with
 *      the code for miniports. miniports and minisockets should coexist.
 */

#include "network.h"
#include "minimsg.h"

typedef struct minisocket* minisocket_t;
typedef enum minisocket_error minisocket_error;

struct minisocket {
  int dummy; /* delete this field */
  /* put your definition of minisockets here */
};

enum minisocket_error {
  SOCKET_NOERROR=0,
  SOCKET_NOMOREPORTS, /* ran out of free ports */
  SOCKET_PORTINUSE, /* server tried to use a port that is already in use */
  SOCKET_NOSERVER, /* client tried to connect to a port without a server */
  SOCKET_BUSY,     /* client tried to connect to a port that is in use */
  SOCKET_SENDERROR,
  SOCKET_RECEIVEERROR
};
  

/* 
 * Listen for a connection from somebody else. When communication link is created
 * return a minisocket_t through which the communication can be made from now on.
 *
 * The argument "port" is the port number on the local machine on which the communication
 * is made. 
 *
 * Return value: the minisocket_t created, otherwise sets it to NULL and returns the errocode
 * in the "error" variable.
 */
minisocket_t minisocket_server_create(int port, minisocket_error *error);

/*
 * Initiate the communication with a remote site. When communication is established create a
 * minisocket through which the communication can be made from now on.
 *
 * The first argument is the network address of the remote machine. 
 *
 * The argument "port" is the port number on the remote machine to which the connection 
 * is made. The port number of the local machine is one of the free port numbers.
 *
 * Return value: the minisocket_t created, otherwise sets it to NULL and returns the errocode
 * in the "error" variable.
 */
minisocket_t minisocket_client_create(network_address_t addr, int port, minisocket_error *error);

/* 
 * Send a message to the other end of the socket. The message on the other end should 
 * be a prefix of the message transmitted (possibly all of it).
 * minisocket_send shluld block until the whole message is reliably transmitted 
 * or an error/timeout occurs
 *
 * Arguments: the socket on which the communication is made (socket), the message
 *            to be transmitted (msg) and its length (len).
 * Return value: returns the number of successfully transmitted bytes. Sets the error
 *            code and returns -1 if an error is encountered.
 */
int minisocket_send(minisocket_t socket, minimsg_t msg, int len, minisocket_error *error);

/*
 * Receive a message from the other end of the socket. Blocks until max_len bytes or a full 
 * message is received (that can be smaller than max_len bytes).
 *
 * Arguments: the socket on which the communication is made (socket), the memory location
 *            where the received message is returned (msg) and its maximum length (max_len).
 * Return value: -1 in case of error and sets the error code, the number of bytes received otherwise
 */
int minisocket_receive(minisocket_t socket, minimsg_t msg, int max_len, minisocket_error *error);

/* Close a connection. If minisocket_close is issued, any send or receive shuld fail.
 * As soon as the other side knows about the close, it shuld fail any send or receive 
 * in progress. The minisocket is destroyed by minisocket_close function.
 * The function should never fail.
 */
void minisocket_close(minisocket_t socket); 

#endif /* __MINISOCKETS_H_ */
