#ifndef __CLOCK_H_
#define __CLOCK_H_
/*
   clock.h

   If you wish to incorporate preemption into your project, you will
   need the following routines. Interrupts will not be enabled unless you
   call minithread_clock_init and provide a handler.

   You should not need to modify this file.
*/

#include <time.h>
#include <signal.h>

/* ClockHandler is a ptr to a function such as "void clkhndlr(void)". */
typedef void (*ClockHandler)();

/* PERIOD is the clock period in microseconds.  It is the interval at
   which clock ticks will be sent
*/

#define SECOND 1000000
#define MILLISECOND 1000

#define	PERIOD 50*MILLISECOND

/* Interrupt level */
typedef int interrupt_level_t;
extern interrupt_level_t interrupt_level; /* this is defined in clock.c */

#define ENABLED 1
#define DISABLED 0


/* set the interrupt level to newlevel, return the old interrupt level

   rather than downgrading the interrupt level to a particular level without
   reference to the old value, you should generally use a set-and-restore
   scheme, as follows:

           interrupt_level_t l;
	   ...
	   l = set_interrupt_level(DISABLED);
	   ... [protected code]
	   set_interrupt_level(ENABLED);

   this way, you are protected against nested interrupt downgrades (i.e. 
   function A disables interrupts and calls B, which also disables them. If
   B enables them, instead of setting the interrupt_level to its old value, 
   interrupts will be enabled when B terminates, when A expected them to be
   disabled. 

   additionally, in the interests of not distorting the period of the timer
   too much, you should try and minimise the length of time interrupts are
   disabled.   
*/
extern interrupt_level_t set_interrupt_level(interrupt_level_t newlevel);


/* dummy routines to prevent unwanted preemption: when a timer event occurs,
   we only preempt if the minithread which is running is at an address between
   start() and end(), which enclose all the minithread and user-supplied code.
   In this way we protect the NT-supplied libraries, which are not
   "minithread-safe" (though they are NT-thread-safe").
*/
extern int start();
extern int end();


/* minithread_clock_init
 * installs your clock interrupt service routine h.
 * h will be called every PERIOD microseconds (defined above).
 * interrupts are disabled by default.  After you enable interrupts
 * then your handler will be called automatically on every clock tick.
 */
extern void minithread_clock_init(ClockHandler h);

#endif  __CLOCK_H_
