package edu.cornell.cs.cs4120.eth.etac.tester;

import edu.cornell.cs.cs4120.eth.SourceFileTest;

import java.io.File;
import java.io.PrintStream;
import java.util.List;

public class EtacOptTester extends EtacCodeGenTester {

    protected String opt;
    protected boolean irChanged = false;

    protected long elapsed, elapsedOpt, elapsedNoOpt;

    public EtacOptTester(String opt) {
        this.opt = opt;
    }

    @Override
    public boolean renameReferenceFiles(SourceFileTest t, File destDir) {
        throw new UnsupportedOperationException();
    }

    @Override
    public boolean normalizeReferenceFiles(SourceFileTest t, File destDir) {
        if (opt == null) return true;

        boolean okay = true;

        if (!opt.equals("reg") && !opt.equals("mc")) {
            String destDirname = appendDirSep(destDir.getPath());
            for (List<String> compilationUnit : t.getSourceFileNames())
                for (String filename : compilationUnit) {
                    String initialIRFilename = initial_ir_Filename(filename);
                    String finalIRFilename = final_ir_Filename(filename);
                    File initialIRFile = new File(destDirname + initialIRFilename);
                    File finalIRFile = new File(destDirname + finalIRFilename);
                    // Compare initial and final IR, which should be the same.
                    okay = compareFiles(initialIRFile, finalIRFile, t) && okay;
                    String initialIRNooptFilename = initial_irnoopt_Filename(filename);
                    String finalIRNooptFilename = final_irnoopt_Filename(filename);
                    // Rename destDir/path/to/file_initial.ir to
                    // destDir/path/to/file_initial.irnoopt.
                    okay =
                            okay
                                    && renameFile(
                                            destDirname + initialIRFilename,
                                            destDirname + initialIRNooptFilename,
                                            t);
                    // Rename destDir/path/to/file_final.ir to destDir/path/to/file_final.irnoopt.
                    okay =
                            okay
                                    && renameFile(
                                            destDirname + finalIRFilename,
                                            destDirname + finalIRNooptFilename,
                                            t);
                }
        }

        // File path/to/file.eta results in destDir/path/to/file.s.
        // Compile destDir/path/to/file.s to destDir/path/to/file.
        File exeFile = prepareExecutable(t, destDir);
        if (exeFile == null) okay = false;
        else {
            String destDirname = appendDirSep(destDir.getPath());
            String exeFilename = exeFile.getName();
            String s_noopt_nmlFilename = normalizedFilename(s_noopt_Filename(exeFilename));
            File s_noopt_nmlFile = new File(destDirname + s_noopt_nmlFilename);
            // Execute destDir/path/to/file
            // and store result in destDir/path/to/file.snoopt.nml.
            okay = normalize(exeFile, s_noopt_nmlFile, t, destDir);
            elapsedNoOpt = elapsed;

            // Compare file.snoopt.nml with file.ssol.nml.
            String ssolnmlFilename = normalizedFilename(ssolFilename(exeFilename));
            File ssolnmlFile = new File(destDirname + ssolnmlFilename);
            if (!compareFiles(ssolnmlFile, s_noopt_nmlFile, t)) okay = false;

            // Rename destDir/path/to/file.s to destDir/path/to/file.snoopt.
            for (List<String> compilationUnit : t.getSourceFileNames())
                for (String filename : compilationUnit) {
                    String sFilename = sFilename(filename);
                    String s_noopt_Filename = s_noopt_Filename(filename);
                    okay =
                            okay
                                    && renameFile(
                                            destDirname + sFilename,
                                            destDirname + s_noopt_Filename,
                                            t);
                }
            // Rename destDir/path/to/file to destDir/path/to/file.noopt.
            String noopt_Filename = noopt_Filename(exeFilename);
            okay = okay && renameFile(destDirname + exeFilename, destDirname + noopt_Filename, t);
        }
        return okay;
    }

    @Override
    public boolean normalizeGeneratedFiles(SourceFileTest t, File destDir) {
        if (opt == null) {
            // Display list of available optimizations.
            t.appendNotice("Available optimizations:");
            String optList = t.getCompilerStdout();
            if (optList != null) t.appendNotice(optList);
            return true;
        }
        boolean okay = super.normalizeGeneratedFiles(t, destDir);
        elapsedOpt = elapsed;
        return okay;
    }

    @Override
    protected int invokeExecutable(File exeFile, PrintStream ps, SourceFileTest t, File destDir) {
        // Start timer for optimization.
        long start = System.nanoTime();
        int result = super.invokeExecutable(exeFile, ps, t, destDir);
        // End timer for optimization.
        long finish = System.nanoTime();
        elapsed = finish - start;
        return result;
    }

    @Override
    public boolean checkResult(SourceFileTest t, File destDir) {
        if (opt != null) {
            if (!opt.equals("reg") && !opt.equals("mc")) {
                String destDirname = appendDirSep(destDir.getPath());
                for (List<String> compilationUnit : t.getSourceFileNames())
                    for (String filename : compilationUnit) {
                        String initialIRFilename = initial_ir_Filename(filename);
                        String finalIRFilename = final_ir_Filename(filename);
                        File initialIRFile = new File(destDirname + initialIRFilename);
                        File finalIRFile = new File(destDirname + finalIRFilename);
                        // Compare initial and final IR.
                        boolean cmpResult = compareFiles(initialIRFile, finalIRFile, t, false);
                        if (!cmpResult) {
                            // Record that optimization has changed the IR.
                            irChanged = true;
                        }
                    }
            }
            // Report speedup.
            double speedup = elapsedNoOpt * 1.0 / elapsedOpt;
            t.appendNotice(
                    String.format(
                            "Speedup: %.3fx (%.3f ms / %.3f ms)",
                            speedup, elapsedNoOpt / 1e6, elapsedOpt / 1e6));
        }
        return super.checkResult(t, destDir);
    }

    @Override
    public boolean cleanupGeneratedFiles(SourceFileTest t, File destDir, File saveDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                // _initial.irnoopt
                String initialIRNooptFilename = initial_irnoopt_Filename(filename);
                okay = moveFileIfExists(destDirname + initialIRNooptFilename, saveDir, t) && okay;
                // _final.irnoopt
                String finalIRNooptFilename = final_irnoopt_Filename(filename);
                okay = moveFileIfExists(destDirname + finalIRNooptFilename, saveDir, t) && okay;
                // _initial.ir
                String initialIRFilename = initial_ir_Filename(filename);
                okay = moveFileIfExists(destDirname + initialIRFilename, saveDir, t) && okay;
                // _final.ir
                String finalIRFilename = final_ir_Filename(filename);
                okay = moveFileIfExists(destDirname + finalIRFilename, saveDir, t) && okay;
                // .noopt executable
                String noOptFilename = noopt_Filename(filename);
                okay = moveFileIfExists(destDirname + noOptFilename, saveDir, t) && okay;
                // .snoopt
                String sNoOptFilename = s_noopt_Filename(filename);
                okay = moveFileIfExists(destDirname + sNoOptFilename, saveDir, t) && okay;
                // .snoopt.nml
                String sNoOptnmlFilename = normalizedFilename(sNoOptFilename);
                okay = okay && moveFileIfExists(destDirname + sNoOptnmlFilename, saveDir, t);
            }
        return super.cleanupGeneratedFiles(t, destDir, saveDir) && okay;
    }

    @Override
    public void getSummary(StringBuffer sb) {
        super.getSummary(sb);
        if (opt != null && !opt.equals("reg") && !opt.equals("mc"))
            sb.append("\nIR modified after optimization: " + irChanged);
    }

    protected String initial_ir_Filename(String filename) {
        return filenameNoExt(filename) + "_initial.ir";
    }

    protected String final_ir_Filename(String filename) {
        return filenameNoExt(filename) + "_final.ir";
    }

    protected String initial_irnoopt_Filename(String filename) {
        return filenameNoExt(filename) + "_initial.irnoopt";
    }

    protected String final_irnoopt_Filename(String filename) {
        return filenameNoExt(filename) + "_final.irnoopt";
    }

    protected String noopt_Filename(String filename) {
        return filenameNoExt(filename) + ".noopt";
    }

    protected String s_noopt_Filename(String filename) {
        return filenameNoExt(filename) + ".snoopt";
    }
}
