package edu.cornell.cs.cs4120.eth.etac;

import edu.cornell.cs.cs4120.eth.AbstractTestDriver;
import edu.cornell.cs.cs4120.eth.FormattedOutput;
import edu.cornell.cs.cs4120.eth.SourceFileTest;
import edu.cornell.cs.cs4120.eth.SourceFileTestCollection;
import edu.cornell.cs.cs4120.eth.etac.tester.EtacCodeGenTester;
import edu.cornell.cs.cs4120.eth.etac.tester.EtacIRGenTester;
import edu.cornell.cs.cs4120.eth.etac.tester.EtacLexingTester;
import edu.cornell.cs.cs4120.eth.etac.tester.EtacOptTester;
import edu.cornell.cs.cs4120.eth.etac.tester.EtacParsingTester;
import edu.cornell.cs.cs4120.eth.etac.tester.EtacTester;
import edu.cornell.cs.cs4120.eth.etac.tester.EtacTypeCheckingTester;

import java.io.File;
import java.util.LinkedList;
import java.util.List;

public class EtacTestDriver extends AbstractTestDriver {

    protected List<EtacTester> testers = new LinkedList<>();

    protected File testDir;
    protected File destDir;

    public EtacTestDriver(SourceFileTestCollection sftc) {
        super(sftc);
    }

    @Override
    public boolean shouldExecute() {
        return true;
    }

    public String commandName() {
        return "etac";
    }

    public void addLexingTest() {
        testers.add(new EtacLexingTester());
    }

    public void addParsingTest() {
        testers.add(new EtacParsingTester());
    }

    public void addTypeCheckingTest() {
        testers.add(new EtacTypeCheckingTester());
    }

    public void addIRGenTest(boolean optimized) {
        testers.add(new EtacIRGenTester(optimized));
    }

    public void addCodeGenTest() {
        testers.add(new EtacCodeGenTester());
    }

    public void addReportOptsTest() {
        testers.add(new EtacOptTester(null));
    }

    protected File getTestDir() {
        if (testDir == null) {
            String testDirname = getPathFromFlagMap("testpath");
            testDir = new File(testDirname);
        }
        return testDir;
    }

    protected File getDestDir() {
        if (destDir == null) {
            String destDirname = getPathFromFlagMap("workpath", "D");
            destDir = new File(destDirname);
        }
        return destDir;
    }

    @Override
    public boolean preTest(SourceFileTestCollection sftc) {
        return true;
    }

    @Override
    public boolean preTest(SourceFileTest t) {
        boolean okay = true;
        File destDir = getDestDir();
        for (EtacTester tester : testers) {
            // Prepare reference files.
            okay = okay && prepareReferenceFiles(tester, t);
            okay = okay && tester.normalizeReferenceFiles(t, destDir);
        }
        return okay;
    }

    protected boolean prepareReferenceFiles(EtacTester tester, SourceFileTest t) {
        // Copy reference files from testDir/file to destDir,
        // unless testDir and destDir are the same.
        File testDir = getTestDir();
        File destDir = getDestDir();
        if (!isSameDirectory(testDir, destDir))
            return tester.copyReferenceFiles(t, testDir, destDir);
        return true;
    }

    @Override
    public boolean postTest(SourceFileTest t) {
        boolean okay = true;
        File destDir = getDestDir();
        for (EtacTester tester : testers) {
            // Normalize generated files.
            if (!tester.normalizeGeneratedFiles(t, destDir)) okay = false;
            else {
                // Compare generated files with solution and report results.
                okay = okay && tester.checkResult(t, destDir);
            }
        }
        return okay;
    }

    @Override
    public boolean cleanup(SourceFileTest t, File saveDir) {
        boolean okay = true;
        File destDir = getDestDir();
        for (EtacTester tester : testers) {
            if (shouldCleanupReferenceFiles()) {
                // Clean up reference files.
                okay = tester.cleanupReferenceFiles(t, destDir, saveDir) && okay;
            }
            // Clean up generated files.
            okay = tester.cleanupGeneratedFiles(t, destDir, saveDir) && okay;
        }
        return okay;
    }

    protected boolean shouldCleanupReferenceFiles() {
        File testDir = getTestDir();
        File destDir = getDestDir();
        // Clean up unless testDir and destDir are the same.
        return !isSameDirectory(testDir, destDir);
    }

    @Override
    public void printTestResult(SourceFileTest t, FormattedOutput pr) {
        File destDir = getDestDir();
        for (EtacTester tester : testers) tester.printTestResult(t, destDir, pr);
    }

    @Override
    public void getSummary(StringBuffer sb) {
        for (EtacTester tester : testers) tester.getSummary(sb);
    }
}
