package edu.cornell.cs.cs4120.eth.etac;

import edu.cornell.cs.cs4120.eth.SourceFileTest;
import edu.cornell.cs.cs4120.eth.SourceFileTestCollection;

import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

public class EtacSourceFileTestCollection extends SourceFileTestCollection {

    public EtacSourceFileTestCollection(
            String testCommand,
            String name,
            String testDir,
            String args,
            List<SourceFileTest> tests) {
        super(testCommand, name, testDir, args, tests);
    }

    @Override
    protected void populatePathFlags() {
        pathFlags.put("-sourcepath", "sourcepath");
        pathFlags.put("-D", "D");
    }

    @Override
    protected List<String> buildCmdLine() {
        List<String> args = super.buildCmdLine();
        List<String> newArgs = new ArrayList<>(args.size());
        for (String arg : args) {
            if (arg.contains("$(testpath")) {
                String workpath = getPathFromFlagMap("workpath");
                String testpath = getPathFromFlagMap("testpath");
                arg = arg.replace("$(testpath)", testpath);
                // Relativize testpath against workpath.
                File testpathFile = new File(arg);
                Path testPath = testpathFile.toPath().normalize();
                if (!testPath.isAbsolute()) {
                    File workpathFile = new File(workpath);
                    Path workPath = workpathFile.toPath().normalize();
                    if (workPath.isAbsolute()) {
                        arg = workpathFile.toURI().relativize(testpathFile.toURI()).getPath();
                    } else arg = workPath.relativize(testPath).toString();
                }
            }
            newArgs.add(arg);
        }
        return newArgs;
    }

    @Override
    protected EtacTestDriver createTestDriver() {
        EtacTestDriver etd;
        switch (testCommand) {
            case "etac":
                etd = new EtacTestDriver(this);
                break;
            case "etac-ref-pre":
                etd = new EtacRefTestDriver(this, true);
                break;
            case "etac-ref":
                etd = new EtacRefTestDriver(this);
                break;
            case "etac-opt":
                etd = new EtacOptTestDriver(this, null);
                break;
            case "etac-ext":
                etd = new EtacExtTestDriver(this);
                break;
            default:
                if (testCommand.startsWith("etac-opt-")) {
                    String opt = testCommand.substring(9);
                    etd = new EtacOptTestDriver(this, opt);
                    break;
                }
                throw new IllegalArgumentException("Unrecognized test command: " + testCommand);
        }
        if (cmdLineHdr.contains("--lex")) etd.addLexingTest();
        if (cmdLineHdr.contains("--parse")) etd.addParsingTest();
        if (cmdLineHdr.contains("--typecheck")) etd.addTypeCheckingTest();
        if (cmdLineHdr.contains("--irgen")) etd.addIRGenTest(!cmdLineHdr.contains("-O"));
        if (cmdLineHdr.contains("-target")) etd.addCodeGenTest();
        if (cmdLineHdr.contains("--report-opts")) etd.addReportOptsTest();
        return etd;
    }
}
