package edu.cornell.cs.cs4120.xth.xic.tester;

import edu.cornell.cs.cs4120.xth.SourceFileTest;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.ListIterator;
import java.util.function.BiPredicate;

public class XicLexingTester extends AbstractXicTester {

    @Override
    public boolean copyReferenceFiles(SourceFileTest t, File testDir, File destDir) {
        boolean okay = true;
        String testDirname = appendDirSep(testDir.getPath());
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                String lexedsolFilename = lexedsolFilename(filename);
                okay =
                        okay
                                && copyFile(
                                        testDirname + lexedsolFilename,
                                        destDirname + lexedsolFilename,
                                        t);
            }
        return okay;
    }

    @Override
    public boolean renameReferenceFiles(SourceFileTest t, File destDir) {
        boolean okay = true;
        // File path/to/file.xi results in destDir/path/to/file.lexed.
        // Rename destDir/path/to/file.lexed to destDir/path/to/file.lexedsol.
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                String lexedFilename = lexedFilename(filename);
                String lexedsolFilename = lexedsolFilename(filename);
                okay =
                        okay
                                && renameFile(
                                        destDirname + lexedFilename,
                                        destDirname + lexedsolFilename,
                                        t);
            }
        return okay;
    }

    @Override
    public boolean normalizeReferenceFiles(SourceFileTest t, File destDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> list : t.getSourceFileNames()) {
            for (String filename : list) {
                String lexedsolFilename = lexedsolFilename(filename);
                // Check existence of files.
                File lexedsolFile = new File(destDirname + lexedsolFilename);
                if (!ensureFileExists(lexedsolFile, t)) {
                    okay = false;
                    continue;
                }
                okay = okay && normalize(lexedsolFile, t);
            }
        }
        return okay;
    }

    @Override
    public boolean normalizeGeneratedFiles(SourceFileTest t, File destDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> list : t.getSourceFileNames()) {
            for (String filename : list) {
                String lexedFilename = lexedFilename(filename);
                // Check existence of files.
                File lexedFile = new File(destDirname + lexedFilename);
                if (!ensureFileExists(lexedFile, t)) {
                    okay = false;
                    continue;
                }
                okay = okay && normalize(lexedFile, t);
            }
        }
        return okay;
    }

    protected boolean normalize(File lexedFile, SourceFileTest t) {
        File nmlLexedFile = new File(normalizedFilename(lexedFile.getPath()));
        try (FileReader lexedfr = new FileReader(lexedFile);
                BufferedReader lexedbr = new BufferedReader(lexedfr);
                FileWriter lexedwr = new FileWriter(nmlLexedFile);
                BufferedWriter lexedbw = new BufferedWriter(lexedwr); ) {
            for (String lexedLine = lexedbr.readLine();
                    lexedLine != null;
                    lexedLine = lexedbr.readLine()) {
                String[] lexed = lexedLine.split(" ", 2);
                // Check array lengths.
                if (lexed.length != 2) {
                    t.appendFailureMessage("Incorrect output format: " + lexedLine);
                    return false;
                }

                // Strip error message.
                if (lexed[1].startsWith("error:")) lexedbw.write(lexed[0] + " error:");
                else lexedbw.write(lexedLine);
                lexedbw.write("\n");
            }
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return false;
        }
        return true;
    }

    protected String normalizedFilename(String filename) {
        return filename + ".nml";
    }

    private List<String> getPermutations(String input) {
        List<String> permutations = new ArrayList<>();
        permutations.add(input);

        ListIterator<String> listerator;

        // normalize escaped double quote
        listerator = permutations.listIterator();
        while (listerator.hasNext()) {
            String s = listerator.next();
            listerator.add(s.replace("\\\"", "\""));
        }

        // normalize escaped single quote
        listerator = permutations.listIterator();
        while (listerator.hasNext()) {
            String s = listerator.next();
            listerator.add(s.replace("\\'", "'"));
        }

        // normalize boolean true / false
        listerator = permutations.listIterator();
        while (listerator.hasNext()) {
            String s = listerator.next();
            String[] split = s.split(" ");
            if (split.length < 2) continue;

            if (split[1].trim().equals("true")) {
                listerator.add(split[0] + " " + "boolean true");
            }
            if (split[1].trim().equals("false")) {
                listerator.add(split[0] + " " + "boolean false");
            }
        }

        return permutations;
    }

    @Override
    public boolean checkResult(SourceFileTest t, File destDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> list : t.getSourceFileNames()) {
            for (String filename : list) {
                String lexedFilename = normalizedFilename(lexedFilename(filename));
                String lexedsolFilename = normalizedFilename(lexedsolFilename(filename));
                File lexedFile = new File(destDirname + lexedFilename);
                File lexedsolFile = new File(destDirname + lexedsolFilename);
                BiPredicate<String, String> comp = (ref, gen) -> getPermutations(ref).contains(gen);
                okay = okay && compareFiles(lexedsolFile, lexedFile, t, true, comp);
            }
        }
        return okay;
    }

    @Override
    public boolean cleanupReferenceFiles(SourceFileTest t, File destDir, File saveDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                // .lexedsol
                String lexedsolFilename = lexedsolFilename(filename);
                okay = okay && moveFileIfExists(destDirname + lexedsolFilename, saveDir, t);
            }
        return okay;
    }

    @Override
    public boolean cleanupGeneratedFiles(SourceFileTest t, File destDir, File saveDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                // .lexed
                String lexedFilename = lexedFilename(filename);
                okay = okay && moveFileIfExists(destDirname + lexedFilename, saveDir, t);
                // .lexed.nml
                String lexednmlFilename = normalizedFilename(lexedFilename);
                okay = okay && moveFileIfExists(destDirname + lexednmlFilename, saveDir, t);
                // .lexedsol.nml
                String lexedsolFilename = lexedsolFilename(filename);
                String lexedsolnmlFilename = normalizedFilename(lexedsolFilename);
                okay = okay && moveFileIfExists(destDirname + lexedsolnmlFilename, saveDir, t);
            }
        return okay;
    }

    protected String lexedFilename(String filename) {
        return filenameNoExt(filename) + ".lexed";
    }

    protected String lexedsolFilename(String filename) {
        return filenameNoExt(filename) + ".lexedsol";
    }
}
