package edu.cornell.cs.cs4120.xth.xic.tester;

import edu.cornell.cs.cs4120.xth.FormattedOutput;
import edu.cornell.cs.cs4120.xth.SourceFileTest;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.PrintStream;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeUnit;

public class XicCodeGenTester extends AbstractXicTester {

    protected static final int TIMEOUT = 30;
    protected static final TimeUnit TIMEUNIT = TimeUnit.SECONDS;

    /** Whether the generated assembly has fancy addressing mode */
    protected boolean hasAddrMode;

    @Override
    public boolean copyReferenceFiles(SourceFileTest t, File testDir, File destDir) {
        boolean okay = true;
        String testDirname = appendDirSep(testDir.getPath());
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                String ssolFilename = normalizedFilename(ssolFilename(filename));
                okay = okay && copyFile(testDirname + ssolFilename, destDirname + ssolFilename, t);
            }
        return okay;
    }

    @Override
    public boolean renameReferenceFiles(SourceFileTest t, File destDir) {
        // File path/to/file.xi results in destDir/path/to/file.s.
        boolean okay = true;

        // Compile destDir/path/to/file.s to destDir/path/to/file.
        File exeFile = prepareExecutable(t, destDir);
        if (exeFile == null) okay = false;
        else {
            String destDirname = appendDirSep(destDir.getPath());
            String ssolnmlFilename = normalizedFilename(ssolFilename(exeFile.getName()));
            File ssolnmlFile = new File(destDirname + ssolnmlFilename);
            // Execute destDir/path/to/file
            // and store result in destDir/path/to/file.ssol.nml,
            okay = normalize(exeFile, ssolnmlFile, t, destDir);

            // Rename destDir/path/to/file.s to destDir/path/to/file.ssol.
            for (List<String> compilationUnit : t.getSourceFileNames())
                for (String filename : compilationUnit) {
                    String sFilename = sFilename(filename);
                    String ssolFilename = ssolFilename(filename);
                    okay =
                            okay
                                    && renameFile(
                                            destDirname + sFilename, destDirname + ssolFilename, t);
                }
            // Rename destDir/path/to/file to destDir/path/to/file.sol.
            String exeFilename = exeFile.getName();
            String solFilename = solFilename(exeFilename);
            okay = okay && renameFile(destDirname + exeFilename, destDirname + solFilename, t);
        }
        return okay;
    }

    @Override
    public boolean normalizeReferenceFiles(SourceFileTest t, File destDir) {
        // Already normalized.
        // Execution was done as part of renaming.
        return true;
    }

    @Override
    public boolean normalizeGeneratedFiles(SourceFileTest t, File destDir) {
        File exeFile = prepareExecutable(t, destDir);
        if (exeFile == null) return false;
        String destDirname = appendDirSep(destDir.getPath());
        String sFilename = sFilename(exeFile.getName());
        String snmlFilename = normalizedFilename(sFilename);
        File snmlFile = new File(destDirname + snmlFilename);
        checkAddressingMode(t, destDir);
        return normalize(exeFile, snmlFile, t, destDir);
    }

    protected File prepareExecutable(SourceFileTest t, File destDir) {
        List<String> sFilenames = new LinkedList<>();
        String exeFilename = null;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> list : t.getSourceFileNames()) {
            for (String filename : list) {
                String sFilename = sFilename(filename);
                // Check existence of files.
                File sFile = new File(destDirname + sFilename);
                if (!ensureFileExists(sFile, t)) {
                    return null;
                }
                sFilenames.add(sFile.getAbsolutePath());
                if (exeFilename == null) exeFilename = filenameNoExt(filename);
            }
        }

        int ret = invokeLinker(sFilenames, exeFilename, t, destDir);
        if (ret != 0) {
            if (ret > 0) t.appendFailureMessage("Failed to link: linkxi.sh exit code " + ret);
            return null;
        }
        // Check existence of executable.
        File exeFile = new File(destDirname + exeFilename);
        if (!ensureFileExists(exeFile, t)) return null;
        return exeFile;
    }

    protected int invokeLinker(
            List<String> cmdLine, String exeFilename, SourceFileTest t, File destDir) {
        String linkerDirname = "runtime/";
        String linkerCmd = "";
        File linkerDir = new File(linkerDirname);
        if (!linkerDir.isAbsolute()) {
            Path wd = Paths.get(".").toAbsolutePath();
            linkerCmd += wd.toString() + File.separator;
        }
        linkerCmd += linkerDirname + "linkxi.sh";
        // Check existence of linker command
        File linker = new File(linkerCmd);
        if (!linker.isFile()) {
            t.appendFailureMessage("File " + linkerCmd + " does not exist.");
            return -3;
        }

        cmdLine.add(0, linkerCmd);
        cmdLine.add("-o");
        cmdLine.add(exeFilename);
        ProcessBuilder pb = new ProcessBuilder(cmdLine);
        // Invoke linker in work/path.
        File workDir = destDir;
        pb.directory(workDir);
        try {
            StringBuffer sbout, sberr;
            Process p = pb.start();
            if (!p.waitFor(TIMEOUT, TIMEUNIT)) {
                t.appendFailureMessage(
                        "Linker time limit of " + TIMEOUT + " " + TIMEUNIT + " exceeded");
                p.destroyForcibly();
            }
            InputStream outStream = p.getInputStream(); // normal output of the shell
            InputStream errStream = p.getErrorStream(); // error output of the shell
            try (InputStreamReader or = new InputStreamReader(outStream);
                    InputStreamReader er = new InputStreamReader(errStream);
                    BufferedReader osr = new BufferedReader(or);
                    BufferedReader esr = new BufferedReader(er)) {
                String line = null;
                sbout = new StringBuffer();
                while ((line = osr.readLine()) != null) {
                    sbout.append(line);
                    sbout.append('\n');
                }
                sberr = new StringBuffer();
                while ((line = esr.readLine()) != null) {
                    sberr.append(line);
                    sberr.append('\n');
                }
            }
            if (sbout.length() > 0) {
                t.appendNotice("[Linker's standard output:");
                t.appendNotice(sbout.toString());
                t.appendNotice("]");
            }
            if (sberr.length() > 0) {
                t.appendNotice("[Linker's standard error:");
                t.appendNotice(sberr.toString());
                t.appendNotice("]");
            }
            return p.exitValue();
        } catch (InterruptedException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return -1;
        } catch (IOException e) {
            t.appendFailureMessage(e.getMessage());
            return -2;
        }
    }

    protected void checkAddressingMode(SourceFileTest t, File destDir) {
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> list : t.getSourceFileNames()) {
            for (String filename : list) {
                String sFilename = sFilename(filename);
                File sFile = new File(destDirname + sFilename);
                try (FileReader sfr = new FileReader(sFile);
                        BufferedReader br = new BufferedReader(sfr); ) {
                    for (String line = br.readLine(); line != null; line = br.readLine()) {
                        if (line.matches(".*\\d+\\(%\\w{3}(, %\\w{3})?(, \\d+)?\\).*")) {
                            hasAddrMode = true;
                            return;
                        }
                    }
                } catch (IOException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
            }
        }
    }

    protected boolean normalize(File exeFile, File snmlFile, SourceFileTest t, File destDir) {
        File snmlDir = snmlFile.getParentFile();
        if (!snmlDir.exists()) snmlDir.mkdirs();
        try (PrintStream ps = new PrintStream(snmlFile); ) {
            int ret = invokeExecutable(exeFile, ps, t, destDir);
            if (ret != 0) {
                if (ret == -3) {
                    ps.println("_xi_out_of_bounds called");
                    return true;
                } else if (ret > 0)
                    t.appendFailureMessage(
                            "Failed to execute: " + exeFile.getName() + " exit code " + ret);
                return false;
            }
        } catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return false;
        }
        return true;
    }

    protected int invokeExecutable(File exeFile, PrintStream ps, SourceFileTest t, File destDir) {
        List<String> cmdLine = Collections.singletonList("./" + exeFile.getName());
        ProcessBuilder pb = new ProcessBuilder(cmdLine);
        // Invoke linker in work/path.
        File workDir = destDir;
        pb.directory(workDir);
        try {
            StringBuffer sberr;
            Process p = pb.start();
            if (!p.waitFor(TIMEOUT, TIMEUNIT)) {
                t.appendFailureMessage(
                        "Executable time limit of " + TIMEOUT + " " + TIMEUNIT + " exceeded");
                p.destroyForcibly();
            }
            InputStream outStream = p.getInputStream(); // normal output of the shell
            InputStream errStream = p.getErrorStream(); // error output of the shell
            try (InputStreamReader or = new InputStreamReader(outStream);
                    InputStreamReader er = new InputStreamReader(errStream);
                    BufferedReader osr = new BufferedReader(or);
                    BufferedReader esr = new BufferedReader(er)) {
                String line = null;
                while ((line = osr.readLine()) != null) {
                    ps.println(line);
                }
                sberr = new StringBuffer();
                while ((line = esr.readLine()) != null) {
                    sberr.append(line);
                    sberr.append('\n');
                }
            }
            if (sberr.length() > 0) {
                if (sberr.toString().equals("Array index out of bounds\n")) return -3;
                t.appendNotice("[Executable's standard error:");
                t.appendNotice(sberr.toString());
                t.appendNotice("]");
            }
            return p.exitValue();
        } catch (InterruptedException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return -1;
        } catch (IOException e) {
            t.appendFailureMessage(e.getMessage());
            return -2;
        }
    }

    protected String normalizedFilename(String filename) {
        return filename + ".nml";
    }

    @Override
    public boolean checkResult(SourceFileTest t, File destDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> list : t.getSourceFileNames()) {
            for (String filename : list) {
                String snmlFilename = normalizedFilename(sFilename(filename));
                String ssolnmlFilename = normalizedFilename(ssolFilename(filename));
                File snmlFile = new File(destDirname + snmlFilename);
                File ssolnmlFile = new File(destDirname + ssolnmlFilename);
                if (!compareFiles(ssolnmlFile, snmlFile, t)) okay = false;
            }
        }
        return okay;
    }

    @Override
    public boolean cleanupReferenceFiles(SourceFileTest t, File destDir, File saveDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                // .ssol.nml
                String ssolFilename = ssolFilename(filename);
                String ssolnmlFilename = normalizedFilename(ssolFilename);
                okay = okay && moveFileIfExists(destDirname + ssolnmlFilename, saveDir, t);
            }
        return okay;
    }

    @Override
    public boolean cleanupGeneratedFiles(SourceFileTest t, File destDir, File saveDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                // executable
                okay = okay && moveFileIfExists(destDirname + filenameNoExt(filename), saveDir, t);
                // .sol
                String solFilename = solFilename(filename);
                okay = okay && moveFileIfExists(destDirname + solFilename, saveDir, t);
                // .s
                String sFilename = sFilename(filename);
                okay = okay && moveFileIfExists(destDirname + sFilename, saveDir, t);
                // .ssol
                String ssolFilename = ssolFilename(filename);
                okay = okay && moveFileIfExists(destDirname + ssolFilename, saveDir, t);
                // .s.nml
                String snmlFilename = normalizedFilename(sFilename);
                okay = okay && moveFileIfExists(destDirname + snmlFilename, saveDir, t);
            }
        return okay;
    }

    @Override
    public void printTestResult(SourceFileTest t, File destDir, FormattedOutput pr) {
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                // .s
                String sFilename = sFilename(filename);
                // .s.nml
                String snmlFilename = normalizedFilename(sFilename);
                pr.printHeader("Generated result:");
                pr.printCode(new File(destDirname + snmlFilename));
                // .ssol.nml
                String ssolnmlFilename = normalizedFilename(ssolFilename(filename));
                pr.printHeader("Expected result:");
                pr.printCode(new File(destDirname + ssolnmlFilename));
            }
    }

    @Override
    public void getSummary(StringBuffer sb) {
        sb.append("\nHas addressing mode: " + hasAddrMode);
    }

    protected String sFilename(String filename) {
        return filenameNoExt(filename) + ".s";
    }

    protected String solFilename(String filename) {
        return filenameNoExt(filename) + ".sol";
    }

    protected String ssolFilename(String filename) {
        return filenameNoExt(filename) + ".ssol";
    }
}
