package edu.cornell.cs.cs4120.xth.xic;

import edu.cornell.cs.cs4120.xth.AbstractTestDriver;
import edu.cornell.cs.cs4120.xth.FormattedOutput;
import edu.cornell.cs.cs4120.xth.SourceFileTest;
import edu.cornell.cs.cs4120.xth.SourceFileTestCollection;
import edu.cornell.cs.cs4120.xth.xic.tester.XicCodeGenTester;
import edu.cornell.cs.cs4120.xth.xic.tester.XicIRGenTester;
import edu.cornell.cs.cs4120.xth.xic.tester.XicLexingTester;
import edu.cornell.cs.cs4120.xth.xic.tester.XicOptTester;
import edu.cornell.cs.cs4120.xth.xic.tester.XicParsingTester;
import edu.cornell.cs.cs4120.xth.xic.tester.XicTester;
import edu.cornell.cs.cs4120.xth.xic.tester.XicTypeCheckingTester;

import java.io.File;
import java.util.LinkedList;
import java.util.List;

public class XicTestDriver extends AbstractTestDriver {

    protected List<XicTester> testers = new LinkedList<>();

    protected File testDir;
    protected File destDir;

    public XicTestDriver(SourceFileTestCollection sftc) {
        super(sftc);
    }

    @Override
    public boolean shouldExecute() {
        return true;
    }

    public String commandName() {
        return "xic";
    }

    public void addLexingTest() {
        testers.add(new XicLexingTester());
    }

    public void addParsingTest() {
        testers.add(new XicParsingTester());
    }

    public void addTypeCheckingTest() {
        testers.add(new XicTypeCheckingTester());
    }

    public void addIRGenTest(boolean optimized) {
        testers.add(new XicIRGenTester(optimized));
    }

    public void addCodeGenTest() {
        testers.add(new XicCodeGenTester());
    }

    public void addReportOptsTest() {
        testers.add(new XicOptTester(null));
    }

    protected File getTestDir() {
        if (testDir == null) {
            String testDirname = getPathFromFlagMap("testpath");
            testDir = new File(testDirname);
        }
        return testDir;
    }

    protected File getDestDir() {
        if (destDir == null) {
            String destDirname = getPathFromFlagMap("workpath", "D");
            destDir = new File(destDirname);
        }
        return destDir;
    }

    @Override
    public boolean preTest(SourceFileTestCollection sftc) {
        return true;
    }

    @Override
    public boolean preTest(SourceFileTest t) {
        boolean okay = true;
        File destDir = getDestDir();
        for (XicTester tester : testers) {
            // Prepare reference files.
            okay = okay && prepareReferenceFiles(tester, t);
            okay = okay && tester.normalizeReferenceFiles(t, destDir);
        }
        return okay;
    }

    protected boolean prepareReferenceFiles(XicTester tester, SourceFileTest t) {
        // Copy reference files from testDir/file to destDir,
        // unless testDir and destDir are the same.
        File testDir = getTestDir();
        File destDir = getDestDir();
        if (!isSameDirectory(testDir, destDir))
            return tester.copyReferenceFiles(t, testDir, destDir);
        return true;
    }

    @Override
    public boolean postTest(SourceFileTest t) {
        boolean okay = true;
        File destDir = getDestDir();
        for (XicTester tester : testers) {
            // Normalize generated files.
            if (!tester.normalizeGeneratedFiles(t, destDir)) okay = false;
            else {
                // Compare generated files with solution and report results.
                okay = okay && tester.checkResult(t, destDir);
            }
        }
        return okay;
    }

    @Override
    public boolean cleanup(SourceFileTest t, File saveDir) {
        boolean okay = true;
        File destDir = getDestDir();
        for (XicTester tester : testers) {
            if (shouldCleanupReferenceFiles()) {
                // Clean up reference files.
                okay = tester.cleanupReferenceFiles(t, destDir, saveDir) && okay;
            }
            // Clean up generated files.
            okay = tester.cleanupGeneratedFiles(t, destDir, saveDir) && okay;
        }
        return okay;
    }

    protected boolean shouldCleanupReferenceFiles() {
        File testDir = getTestDir();
        File destDir = getDestDir();
        // Clean up unless testDir and destDir are the same.
        return !isSameDirectory(testDir, destDir);
    }

    @Override
    public void printTestResult(SourceFileTest t, FormattedOutput pr) {
        File destDir = getDestDir();
        for (XicTester tester : testers) tester.printTestResult(t, destDir, pr);
    }

    @Override
    public void getSummary(StringBuffer sb) {
        for (XicTester tester : testers) tester.getSummary(sb);
    }
}
