package edu.cornell.cs.cs4120.xth.xic;

import edu.cornell.cs.cs4120.xth.SourceFileTest;
import edu.cornell.cs.cs4120.xth.SourceFileTestCollection;

import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

public class XicSourceFileTestCollection extends SourceFileTestCollection {

    public XicSourceFileTestCollection(
            String testCommand,
            String name,
            String testDir,
            String args,
            List<SourceFileTest> tests) {
        super(testCommand, name, testDir, args, tests);
    }

    @Override
    protected void populatePathFlags() {
        pathFlags.put("-sourcepath", "sourcepath");
        pathFlags.put("-D", "D");
    }

    @Override
    protected List<String> buildCmdLine() {
        List<String> args = super.buildCmdLine();
        List<String> newArgs = new ArrayList<>(args.size());
        for (String arg : args) {
            if (arg.contains("$(testpath")) {
                String workpath = getPathFromFlagMap("workpath");
                String testpath = getPathFromFlagMap("testpath");
                arg = arg.replace("$(testpath)", testpath);
                // Relativize testpath against workpath.
                File testpathFile = new File(arg);
                Path testPath = testpathFile.toPath().normalize();
                if (!testPath.isAbsolute()) {
                    File workpathFile = new File(workpath);
                    Path workPath = workpathFile.toPath().normalize();
                    if (workPath.isAbsolute()) {
                        arg = workpathFile.toURI().relativize(testpathFile.toURI()).getPath();
                    } else arg = workPath.relativize(testPath).toString();
                }
            }
            newArgs.add(arg);
        }
        return newArgs;
    }

    @Override
    protected XicTestDriver createTestDriver() {
        XicTestDriver xtd;
        switch (testCommand) {
            case "xic":
                xtd = new XicTestDriver(this);
                break;
            case "xic-ref-pre":
                xtd = new XicRefTestDriver(this, true);
                break;
            case "xic-ref":
                xtd = new XicRefTestDriver(this);
                break;
            case "xic-opt":
                xtd = new XicOptTestDriver(this, null);
                break;
            case "xic-ext":
                xtd = new XicExtTestDriver(this);
                break;
            default:
                if (testCommand.startsWith("xic-opt-")) {
                    String opt = testCommand.substring(8);
                    xtd = new XicOptTestDriver(this, opt);
                    break;
                }
                throw new IllegalArgumentException("Unrecognized test command: " + testCommand);
        }
        if (cmdLineHdr.contains("--lex")) xtd.addLexingTest();
        if (cmdLineHdr.contains("--parse")) xtd.addParsingTest();
        if (cmdLineHdr.contains("--typecheck")) xtd.addTypeCheckingTest();
        if (cmdLineHdr.contains("--irgen")) xtd.addIRGenTest(!cmdLineHdr.contains("-O"));
        if (cmdLineHdr.contains("-target")) xtd.addCodeGenTest();
        if (cmdLineHdr.contains("--report-opts")) xtd.addReportOptsTest();
        return xtd;
    }
}
