package edu.cornell.cs.cs4120.xth;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

public class MarkdownReporter extends FormattedOutput {

    protected int indent = 0;
    protected boolean printIndent = true;
    protected String testCollectionName;
    protected String filename;
    protected StringBuffer txtsb = new StringBuffer();

    public MarkdownReporter(String filename) {
        this.filename = filename;
        txtsb.append("# Summary\n");
    }

    protected void beginBlock() {
        indent += 2;
    }

    protected void endBlock() {
        indent -= 2;
    }

    protected void print(String s) {
        if (printIndent) printIndent();
        txtsb.append(s);
        printIndent = false;
    }

    protected void println(String s) {
        if (printIndent) printIndent();
        txtsb.append(s);
        txtsb.append("\n");
        printIndent = true;
    }

    protected void println() {
        if (printIndent) printIndent();
        txtsb.append("\n");
        printIndent = true;
    }

    protected void printIndent() {
        for (int i = 0; i < indent; i++) txtsb.append(' ');
    }

    protected void startScriptTestSuite(ScriptTestSuite sts) {
        println("Test script: " + sts.getName());
        beginBlock();
    }

    protected void finishScriptTestSuite(ScriptTestSuite sts) {
        endBlock();
        String notice = sts.getNotice();
        if (notice != null) println(notice);

        if (!sts.success() && sts.getFailureMessage() != null) println(sts.getFailureMessage());

        println(
                sts.getName()
                        + ": "
                        + sts.getSuccessfulTestCount()
                        + " out of "
                        + sts.getExecutedTestCount()
                        + " tests succeeded.");
    }

    protected void startSourceFileTestCollection(SourceFileTestCollection sftc) {
        testCollectionName = sftc.getName();
        println("Test collection: " + testCollectionName);
        beginBlock();
    }

    protected void finishSourceFileTestCollection(SourceFileTestCollection sftc) {
        endBlock();
        String notice = sftc.getNotice();
        if (notice != null) println(notice);

        if (!sftc.success() && sftc.getFailureMessage() != null) println(sftc.getFailureMessage());

        println(sftc.getSummary());
    }

    protected void startSourceFileTest(SourceFileTest sft) {
        print(anonymizedName(sft.getName()) + ": ");
        beginBlock();
    }

    protected void finishSourceFileTest(SourceFileTest sft) {
        endBlock();
        String notice = sft.getNotice();
        if (notice != null) print("[" + anonymizedName(notice) + "] ");

        String result;
        if (sft.success()) result = "OK";
        else {
            String msg = sft.getFailureMessage();
            if (msg == null) msg = "Failed (no message)";
            result = anonymizedName(msg);
        }
        println(result);

        if (!sft.success()) {
            txtsb.append(
                    "\n---\n# "
                            + testCollectionName
                            + ": "
                            + anonymizedName(sft.getUniqueId())
                            + "\n");

            sft.printTestResult(this);

            txtsb.append("\n---\n");
        }
    }

    protected void startBuildTest(BuildTest b) {
        print(b.getName() + ": ");
        beginBlock();
    }

    protected void finishBuildTest(BuildTest b) {
        endBlock();
        String notice = b.getNotice();
        if (notice != null) println(notice);

        if (b.success()) println("OK");
        else if (b.getFailureMessage() != null) println(b.getFailureMessage());
        else println("Failed (no message)");
    }

    public void printHeader(String header) {
        txtsb.append("\n## " + header + "\n");
    }

    public void printText(String content) {
        txtsb.append(content);
    }

    public void printCode(File srcFile) {
        try (BufferedReader br = new BufferedReader(new FileReader(srcFile))) {
            List<String> lines = new ArrayList<>();
            txtsb.append("\n```\n");
            for (String line = br.readLine(); line != null; line = br.readLine()) {
                lines.add(line);
                txtsb.append(line);
                txtsb.append("\n");
            }
            txtsb.append("\n```\n");
        } catch (FileNotFoundException e) {
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public void flush() {
        try {
            BufferedWriter writer = new BufferedWriter(new FileWriter(filename));
            writer.write(txtsb.toString());

            writer.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }
}
