package edu.cornell.cs.cs4120.xth;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.regex.Pattern;

/** */
public abstract class AbstractTest implements Test {
    protected String name;
    protected String description;
    protected boolean success = false;
    protected boolean hasRun = false;
    protected TestResult testResult;

    protected StringBuffer noticeBuilder = null;
    protected StringBuffer failureMessageBuilder = null;

    protected List<AbstractOutputController> outputControllers;

    public AbstractTest(String name) {
        this.name = name;
        this.outputControllers = new ArrayList<>();
    }

    @Override
    public void addOutputController(AbstractOutputController output) {
        if (!this.outputControllers.contains(output)) this.outputControllers.add(output);
    }

    @Override
    public final boolean run() {
        preRun();
        success = runTest();

        hasRun = true;
        Date lastSuccess = null;
        if (getTestResult() != null) lastSuccess = getTestResult().dateLastSuccess;
        setTestResult(createTestResult(lastSuccess));
        postRun();
        return success();
    }

    @Override
    public boolean shouldExecute(TestResult tr) {
        if (!matchFilter()) return false;

        if (Main.options.testPreviouslyFailedOnly
                && tr != null
                && tr.dateLastSuccess != null
                && tr.dateLastSuccess.equals(tr.dateTestRun)) return false;
        return true;
    }

    protected boolean matchFilter() {
        if (Main.options.testFilters.isEmpty()) return true;
        for (String testFilter : Main.options.testFilters)
            if (Pattern.matches(testFilter, getName())) return true;
        return false;
    }

    @Override
    public boolean haltOnFailure() {
        return false;
    }

    protected abstract boolean runTest();

    protected void preRun() {
        for (AbstractOutputController output : this.outputControllers) {
            if (output != null) output.startTest(this);
        }
    }

    protected void postRun() {
        for (AbstractOutputController output : this.outputControllers) {
            if (output != null) output.finishTest(this);
        }
    }

    protected TestResult createTestResult(Date lastSuccess) {
        Date lastRun = new Date();
        if (success()) lastSuccess = lastRun;
        return new TestResult(this, lastRun, lastSuccess);
    }

    @Override
    public boolean success() {
        return success;
    }

    @Override
    public int getTotalTestCount() {
        return 1;
    }

    @Override
    public int getExecutedTestCount() {
        return 1;
    }

    @Override
    public int getSuccessfulTestCount() {
        return success() ? 1 : 0;
    }

    @Override
    public String getNotice() {
        return noticeBuilder == null ? null : noticeBuilder.toString();
    }

    @Override
    public String getFailureMessage() {
        return failureMessageBuilder == null ? null : failureMessageBuilder.toString();
    }

    public void appendNotice(String notice) {
        if (noticeBuilder == null) noticeBuilder = new StringBuffer(notice);
        else {
            noticeBuilder.append("\n");
            noticeBuilder.append(notice);
        }
    }

    public void appendFailureMessage(String failureMessage) {
        if (failureMessageBuilder == null) failureMessageBuilder = new StringBuffer(failureMessage);
        else {
            failureMessageBuilder.append("\n");
            failureMessageBuilder.append(failureMessage);
        }
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public String getName() {
        return name;
    }

    public void setDescription(String string) {
        description = string;
    }

    public void setName(String string) {
        name = string;
    }

    @Override
    public TestResult getTestResult() {
        return testResult;
    }

    @Override
    public void setTestResult(TestResult tr) {
        testResult = tr;
    }

    @Override
    public int[] displayTestResult(OutputController outCtrl) {
        int total = 0;
        int lastSuccess = 0;
        int neverRun = 0;
        int neverSuccess = 0;
        TestResult tr = getTestResult();

        total++;
        if (tr != null && tr.dateLastSuccess != null && tr.dateLastSuccess.equals(tr.dateTestRun))
            lastSuccess++;
        if (tr == null || tr.dateTestRun == null) neverRun++;
        if (tr == null || tr.dateLastSuccess == null) neverSuccess++;

        outCtrl.displayTestResults(tr, getName());

        return new int[] {total, lastSuccess, neverRun, neverSuccess};
    }
}
