package edu.cornell.cs.cs4120.xth.xic.tester;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.List;

import edu.cornell.cs.cs4120.xth.PDFReporter;
import edu.cornell.cs.cs4120.xth.SourceFileTest;
import polyglot.util.InternalCompilerError;

public class XicTypeCheckingTester extends AbstractXicTester {

    @Override
    public boolean copyReferenceFiles(SourceFileTest t, File testDir,
            File destDir) {
        boolean okay = true;
        String testDirname = appendDirSep(testDir.getPath());
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                String typedsolFilename = typedsolFilename(filename);
                okay = okay && copyFile(testDirname + typedsolFilename,
                                        destDirname + typedsolFilename,
                                        t);
            }
        return okay;
    }

    @Override
    public boolean renameReferenceFiles(SourceFileTest t, File destDir) {
        boolean okay = true;
        // File path/to/file.xi results in destDir/path/to/file.typed.
        // Rename destDir/path/to/file.typed to destDir/path/to/file.typedsol.
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                String typedFilename = typedFilename(filename);
                String typedsolFilename = typedsolFilename(filename);
                okay = okay && renameFile(destDirname + typedFilename,
                                          destDirname + typedsolFilename,
                                          t);
            }
        return okay;
    }

    @Override
    public boolean normalizeReferenceFiles(SourceFileTest t, File destDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> list : t.getSourceFileNames()) {
            for (String filename : list) {
                String typedsolFilename = typedsolFilename(filename);
                // Check existence of files.
                File typedsolFile = new File(destDirname + typedsolFilename);
                if (!ensureFileExists(typedsolFile, t)) {
                    okay = false;
                    continue;
                }
                okay = okay && normalize(typedsolFile, t);
            }
        }
        return okay;
    }

    @Override
    public boolean normalizeGeneratedFiles(SourceFileTest t, File destDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> list : t.getSourceFileNames()) {
            for (String filename : list) {
                String typedFilename = typedFilename(filename);
                // Check existence of files.
                File typedFile = new File(destDirname + typedFilename);
                if (!ensureFileExists(typedFile, t)) {
                    okay = false;
                    continue;
                }
                okay = okay && normalize(typedFile, t);
            }
        }
        return okay;
    }

    protected boolean normalize(File typedFile, SourceFileTest t) {
        File nmltypedFile = new File(normalizedFilename(typedFile.getPath()));
        try (FileReader typedfr = new FileReader(typedFile);
             BufferedReader typedbr = new BufferedReader(typedfr);
             FileWriter typedfw = new FileWriter(nmltypedFile);
             BufferedWriter bw = new BufferedWriter(typedfw);) {
            for (String typedLine =
                    typedbr.readLine(); typedLine != null; typedLine =
                            typedbr.readLine()) {
                if (typedLine.equals("Valid Xi Program")) {
                    bw.write(typedLine);
                    bw.write("\n");
                }
                else if (typedLine.matches("^\\d+:\\d+ error:.*$")) {
                    String[] typed = typedLine.split(" ", 2);
                    // Check array lengths.
                    if (typed.length != 2) {
                        t.appendFailureMessage("Incorrect output format: "
                                + typedLine);
                        return false;
                    }

                    // Strip position and error message.
                    if (typed[1].startsWith("error:"))
                        bw.write(": error:\n");
                    else {
                        t.appendFailureMessage("Incorrect output format: "
                                + typedLine);
                        return false;
                    }
                }
                else {
                    t.appendFailureMessage("Incorrect output format: "
                            + typedLine);
                    return false;
                }
            }
        }
        catch (InternalCompilerError e) {
            t.appendFailureMessage(e.getMessage());
            return false;
        }
        catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return false;
        }
        return true;
    }

    protected String normalizedFilename(String filename) {
        return filename + ".nml";
    }

    @Override
    public boolean checkResult(SourceFileTest t, File destDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> list : t.getSourceFileNames()) {
            for (String filename : list) {
                String typeddFilename =
                        normalizedFilename(typedFilename(filename));
                String typedsolFilename =
                        normalizedFilename(typedsolFilename(filename));
                File typedFile = new File(destDirname + typeddFilename);
                File typedsolFile = new File(destDirname + typedsolFilename);
                okay = okay && compareFiles(typedsolFile, typedFile, t);
            }
        }
        return okay;
    }

    @Override
    public boolean cleanupReferenceFiles(SourceFileTest t, File destDir,
            File saveDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                // .typedsol
                String typedsolFilename = typedsolFilename(filename);
                okay = okay && moveFileIfExists(destDirname + typedsolFilename,
                                                saveDir,
                                                t);
            }
        return okay;
    }

    @Override
    public boolean cleanupGeneratedFiles(SourceFileTest t, File destDir,
            File saveDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                // .typed
                String typedFilename = typedFilename(filename);
                okay = okay && moveFileIfExists(destDirname + typedFilename,
                                                saveDir,
                                                t);
                // .typed.nml
                String typednmlFilename = normalizedFilename(typedFilename);
                okay = okay && moveFileIfExists(destDirname + typednmlFilename,
                                                saveDir,
                                                t);
                // .typedsol.nml
                String typedsolFilename = typedsolFilename(filename);
                String typedsolnmlFilename =
                        normalizedFilename(typedsolFilename);
                okay = okay
                        && moveFileIfExists(destDirname + typedsolnmlFilename,
                                            saveDir,
                                            t);
            }
        return okay;
    }

    @Override
    public void printTestResult(SourceFileTest t, File destDir,
            PDFReporter pr) {
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                // .typed
                String typedFilename = typedFilename(filename);
                pr.printHeader("Generated result for --typecheck:");
                pr.printCode(new File(destDirname + typedFilename));
                // .typedsol
                String typedsolFilename = typedsolFilename(filename);
                pr.printHeader("Expected result for --typecheck:");
                pr.printCode(new File(destDirname + typedsolFilename));
            }
    }

    protected String typedFilename(String filename) {
        return filenameNoExt(filename) + ".typed";
    }

    protected String typedsolFilename(String filename) {
        return filenameNoExt(filename) + ".typedsol";
    }
}
