package edu.cornell.cs.cs4120.xth.xic.tester;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.util.List;

import edu.cornell.cs.cs4120.xth.PDFReporter;
import edu.cornell.cs.cs4120.xth.SourceFileTest;
import polyglot.util.CodeWriter;
import polyglot.util.InternalCompilerError;
import polyglot.util.OptimalCodeWriter;

public class XicParsingTester extends AbstractXicTester {

    @Override
    public boolean copyReferenceFiles(SourceFileTest t, File testDir,
            File destDir) {
        boolean okay = true;
        String testDirname = appendDirSep(testDir.getPath());
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                String parsedsolFilename = parsedsolFilename(filename);
                okay = okay && copyFile(testDirname + parsedsolFilename,
                                        destDirname + parsedsolFilename,
                                        t);
            }
        return okay;
    }

    @Override
    public boolean renameReferenceFiles(SourceFileTest t, File destDir) {
        boolean okay = true;
        // File path/to/file.xi results in destDir/path/to/file.parsed.
        // Rename destDir/path/to/file.parsed to destDir/path/to/file.parsedsol.
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                String parsedFilename = parsedFilename(filename);
                String parsedsolFilename = parsedsolFilename(filename);
                okay = okay && renameFile(destDirname + parsedFilename,
                                          destDirname + parsedsolFilename,
                                          t);
            }
        return okay;
    }

    @Override
    public boolean normalizeReferenceFiles(SourceFileTest t, File destDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> list : t.getSourceFileNames()) {
            for (String filename : list) {
                String parsedsolFilename = parsedsolFilename(filename);
                // Check existence of files.
                File parsedsolFile = new File(destDirname + parsedsolFilename);
                if (!ensureFileExists(parsedsolFile, t)) {
                    okay = false;
                    continue;
                }
                okay = okay && normalize(parsedsolFile, t);
            }
        }
        return okay;
    }

    @Override
    public boolean normalizeGeneratedFiles(SourceFileTest t, File destDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> list : t.getSourceFileNames()) {
            for (String filename : list) {
                String parsedFilename = parsedFilename(filename);
                // Check existence of files.
                File parsedFile = new File(destDirname + parsedFilename);
                if (!ensureFileExists(parsedFile, t)) {
                    okay = false;
                    continue;
                }
                okay = okay && normalize(parsedFile, t);
            }
        }
        return okay;
    }

    protected boolean normalize(File parsedFile, SourceFileTest t) {
        File nmlparsedFile = new File(normalizedFilename(parsedFile.getPath()));
        try (FileReader parsedfr = new FileReader(parsedFile);
             BufferedReader parsedbr = new BufferedReader(parsedfr);
             FileOutputStream parsedos = new FileOutputStream(nmlparsedFile);
             CodeWriter w = new OptimalCodeWriter(parsedos, 80);) {
            boolean addSpace = false;
            boolean firstLine = true;
            boolean inEscape = false;
            boolean inString = false;
            boolean inChar = false;
            StringBuffer sb = new StringBuffer();
            for (String parsedLine =
                    parsedbr.readLine(); parsedLine != null; parsedLine =
                            parsedbr.readLine()) {
                boolean afterSpace = true;
                if (firstLine) {
                    firstLine = false;
                    if (parsedLine.matches("^\\d+:\\d+ error:.*$")) {
                        String[] parsed = parsedLine.split(" ", 2);
                        // Check array lengths.
                        if (parsed.length != 2) {
                            t.appendFailureMessage("Incorrect output format: "
                                    + parsedLine);
                            return false;
                        }

                        // Strip error message.
                        if (parsed[1].startsWith("error:"))
                            w.write(parsed[0] + " error:");
                        else {
                            t.appendFailureMessage("Incorrect output format: "
                                    + parsedLine);
                            return false;
                        }
                        w.newline();
                        break;
                    }
                }
                for (char c : parsedLine.toCharArray()) {
                    if (!inChar && !inString) {
                        if (c == '(') {
                            w.write(sb.toString());
                            if (addSpace || sb.length() > 0) w.allowBreak(0);
                            sb.setLength(0);
                            addSpace = false;
                            w.begin(1);
                            w.write("(");
                        }
                        else if (c == ')') {
                            w.write(sb.toString());
                            sb.setLength(0);
                            w.write(")");
                            w.end();
                            addSpace = true;
                        }
                        else if (c == ' ') {
                            if (sb.length() > 0) {
                                w.write(sb.toString());
                                sb.setLength(0);
                                addSpace = true;
                            }
                        }
                        else {
                            if (addSpace) {
                                w.allowBreak(0);
                                addSpace = false;
                            }
                            sb.append(c);
                        }
                        afterSpace = c == ' ';
                    }
                    else if (c == '\\')
                        inEscape = true;
                    else {
                        if (addSpace) {
                            w.allowBreak(0);
                            addSpace = false;
                        }
                        if (c == '\'') {
                            if (inEscape) {
                                sb.append("\\'");
                                inEscape = false;
                            }
                            else {
                                sb.append("'");
                                if (inChar)
                                    inChar = false;
                                else if (!inChar && afterSpace) inChar = true;
                            }
                        }
                        else if (c == '"') {
                            if (inEscape) {
                                sb.append("\\\"");
                                inEscape = false;
                            }
                            else {
                                sb.append("\"");
                                inString = !inString;
                            }
                        }
                        else if (inEscape) {
                            sb.append('\\');
                            inEscape = false;
                        }
                        else sb.append(c);
                    }
                }
                if (!inString) {
                    if (sb.length() > 0) {
                        w.write(sb.toString());
                        sb.setLength(0);
                        addSpace = true;
                    }
                }
                else sb.append('\n');
            }
            w.newline();
        }
        catch (InternalCompilerError e) {
            t.appendFailureMessage(e.getMessage());
            return false;
        }
        catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return false;
        }
        return true;
    }

    protected String normalizedFilename(String filename) {
        return filename + ".nml";
    }

    @Override
    public boolean checkResult(SourceFileTest t, File destDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> list : t.getSourceFileNames()) {
            for (String filename : list) {
                String parsedFilename =
                        normalizedFilename(parsedFilename(filename));
                String parsedsolFilename =
                        normalizedFilename(parsedsolFilename(filename));
                File parsedFile = new File(destDirname + parsedFilename);
                File parsedsolFile = new File(destDirname + parsedsolFilename);
                okay = okay && compareFiles(parsedsolFile, parsedFile, t);
            }
        }
        return okay;
    }

    @Override
    public boolean cleanupReferenceFiles(SourceFileTest t, File destDir,
            File saveDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                // .parsedsol
                String parsedsolFilename = parsedsolFilename(filename);
                okay = okay && moveFileIfExists(destDirname + parsedsolFilename,
                                                saveDir,
                                                t);
            }
        return okay;
    }

    @Override
    public boolean cleanupGeneratedFiles(SourceFileTest t, File destDir,
            File saveDir) {
        boolean okay = true;
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                // .parsed
                String parsedFilename = parsedFilename(filename);
                okay = okay && moveFileIfExists(destDirname + parsedFilename,
                                                saveDir,
                                                t);
                // .parsed.nml
                String parsednmlFilename = normalizedFilename(parsedFilename);
                okay = okay && moveFileIfExists(destDirname + parsednmlFilename,
                                                saveDir,
                                                t);
                // .parsedsol.nml
                String parsedsolFilename = parsedsolFilename(filename);
                String parsedsolnmlFilename =
                        normalizedFilename(parsedsolFilename);
                okay = okay
                        && moveFileIfExists(destDirname + parsedsolnmlFilename,
                                            saveDir,
                                            t);
            }
        return okay;
    }

    @Override
    public void printTestResult(SourceFileTest t, File destDir,
            PDFReporter pr) {
        String destDirname = appendDirSep(destDir.getPath());
        for (List<String> compilationUnit : t.getSourceFileNames())
            for (String filename : compilationUnit) {
                // .parsed
                String parsedFilename = parsedFilename(filename);
                pr.printHeader("Generated result for --parse:");
                pr.printCode(new File(destDirname + parsedFilename));
                // .parsedsol
                String parsedsolFilename = parsedsolFilename(filename);
                pr.printHeader("Expected result for --parse:");
                pr.printCode(new File(destDirname + parsedsolFilename));
            }
    }

    protected String parsedFilename(String filename) {
        String extension;
        switch (fileExt(filename).toLowerCase()) {
            case "ixi":
                extension = "iparsed";
                break;
            case "xi":
            default: // TODO probably should report an error
                extension = "parsed";
                break;
        }

        return filenameNoExt(filename) + "." + extension;
    }

    protected String parsedsolFilename(String filename) {
        return parsedFilename(filename) + "sol";
    }
}
