package edu.cornell.cs.cs4120.xth.xic.tester;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.nio.file.Files;
import java.util.function.BiPredicate;

import edu.cornell.cs.cs4120.xth.PDFReporter;
import edu.cornell.cs.cs4120.xth.SourceFileTest;

public abstract class AbstractXicTester implements XicTester {

    protected String appendDirSep(String dirname) {
        if (!dirname.endsWith(File.separator)) dirname += File.separator;
        return dirname;
    }

    protected boolean ensureFileExists(String filepath, SourceFileTest t) {
        return ensureFileExists(new File(filepath), t);
    }

    protected boolean ensureFileExists(File file, SourceFileTest t) {
        if (!file.exists()) {
            String filename = file.getName();
            String dirname = file.getParent();
            t.appendFailureMessage("File " + filename
                    + " does not exist in directory " + dirname);
            return false;
        }
        return true;
    }

    protected boolean ensureFileNotExists(String filepath, SourceFileTest t) {
        return ensureFileNotExists(new File(filepath), t);
    }

    protected boolean ensureFileNotExists(File file, SourceFileTest t) {
        if (file.exists()) {
            String filename = file.getName();
            String dirname = file.getParent();
            t.appendFailureMessage("File " + filename
                    + " already exists in directory " + dirname);
            return false;
        }
        return true;
    }

    protected boolean copyFile(String srcFilepath, String dstFilepath,
            SourceFileTest t) {
        return copyFile(new File(srcFilepath), new File(dstFilepath), t);
    }

    protected boolean copyFile(File srcFile, File dstFile, SourceFileTest t) {
        if (!ensureFileExists(srcFile, t)) return false;
        if (!ensureFileNotExists(dstFile, t)) return false;
        try {
            Files.copy(srcFile.toPath(), dstFile.toPath());
            return true;
        }
        catch (IOException e) {
            // TODO Auto-generated catch block
            e.printStackTrace();
            return false;
        }
    }

    protected boolean renameFile(String srcFilepath, String dstFilepath,
            SourceFileTest t) {
        return renameFile(new File(srcFilepath), new File(dstFilepath), t);
    }

    protected boolean renameFile(File srcFile, File dstFile, SourceFileTest t) {
        if (!ensureFileExists(srcFile, t)) return false;
        if (!ensureFileNotExists(dstFile, t)) return false;
        if (!srcFile.renameTo(dstFile)) {
            String srcFilename = srcFile.getName();
            String srcDirname = srcFile.getParent();
            t.appendFailureMessage("Cannot rename file " + srcFilename
                    + " in directory " + srcDirname);
            return false;
        }
        return true;
    }

    protected boolean deleteFile(String filepath, SourceFileTest t) {
        return deleteFile(new File(filepath), t);
    }

    protected boolean deleteFile(File file, SourceFileTest t) {
        if (file.exists() && !file.delete()) {
            String filename = file.getName();
            String dirname = file.getParent();
            t.appendFailureMessage("Cannot delete file " + filename
                    + " in directory " + dirname);
            return false;
        }
        return true;
    }

    protected boolean moveFileIfExists(String filepath, File dstDir,
            SourceFileTest t) {
        return moveFileIfExists(new File(filepath), dstDir, t);
    }

    protected boolean moveFileIfExists(File file, File dstDir,
            SourceFileTest t) {
        if (file.exists()) return moveFile(file, dstDir, t);
        return true;
    }

    protected boolean moveFile(String filepath, File dstDir, SourceFileTest t) {
        return moveFile(new File(filepath), dstDir, t);
    }

    protected boolean moveFile(File file, File dstDir, SourceFileTest t) {
        if (dstDir == null) return deleteFile(file, t);
        File dstFile = new File(dstDir, file.getName());
        return renameFile(file, dstFile, t);
    }

    protected boolean compareFiles(File expectedFile, File resultFile,
            SourceFileTest t) {
        return compareFiles(expectedFile, resultFile, t, true,
                            (l, r) -> l.equals(r));
    }

    protected boolean compareFiles(File expectedFile, File resultFile,
                                   SourceFileTest t, boolean shouldReport) {
        return compareFiles(expectedFile, resultFile, t, shouldReport,
                            (l, r) -> l.equals(r));
    }

    protected boolean compareFiles(File expectedFile, File resultFile,
                                   SourceFileTest t, boolean shouldReport,
                                   BiPredicate<String, String> lineMatches) {
        // Compare files.
        try (FileReader resultfr = new FileReader(resultFile);
             FileReader expectedfr = new FileReader(expectedFile);
             BufferedReader resultbr = new BufferedReader(resultfr);
             BufferedReader expectedbr = new BufferedReader(expectedfr)) {
            int line = 1;
            String resultLine = resultbr.readLine();
            String expectedLine = expectedbr.readLine();
            for (; resultLine != null
                    && expectedLine != null; line++, resultLine =
                            resultbr.readLine(), expectedLine =
                                    expectedbr.readLine()) {
                if (!lineMatches.test(expectedLine, resultLine)) {
                    if (shouldReport) {
                        t.appendFailureMessage("Mismatch detected at line "
                                + line + " of file " + resultFile.getName());
                        t.appendFailureMessage("expected: " + expectedLine);
                        t.appendFailureMessage("found   : " + resultLine);
                    }
                    return false;
                }
            }
            // Check for mismatched lines.
            if (expectedLine != null) {
                if (shouldReport) t.appendFailureMessage("Missing line in file "
                        + resultFile.getName() + ": " + expectedLine);
                return false;
            }
            if (resultLine != null) {
                if (shouldReport) t.appendFailureMessage("Extra line in file "
                        + resultFile.getName() + ": " + resultLine);
                return false;
            }
        }
        catch (IOException e) {
            t.appendFailureMessage(e.getMessage());
            return false;
        }
        return true;
    }

    protected static String filenameNoExt(String filename) {
        int index = filename.lastIndexOf('.');
        return index == -1 ? filename : filename.substring(0, index);
    }

    protected static String fileExt(String filename) {
        int index = filename.lastIndexOf('.');
        return index == -1 ? "" : filename.substring(index + 1);
    }

    @Override
    public void printTestResult(SourceFileTest t, File destDir,
            PDFReporter pr) {
    }

    @Override
    public void getSummary(StringBuffer sb) {
    }
}
