/**
 An implementation of the Iota 9 standard & runtime library.

 Note: string functions are NOT Unicode clean to avoid dealing with portability
 issues --- ideally, I9 strings should be viewed as UCS-4, but they are treated
 as expanded out local 8-bit encoding
*/

#include "libi9.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
 #include <sys/time.h>
#include "../common/io.h"

#include "../gc6.8/include/gc.h"

/**
 Core runtime
*/

int* I9(_alloc_i)(int size) {
    return GC_malloc(size);
}

void i9_registerFinalizer(void* object, Finalizer* fin) {
    GC_register_finalizer_ignore_self(object, fin, 0, 0, 0);
}

void I9(_outOfBounds_p)(void) {
    fprintf(stderr, "Array index out of bounds\n");
    abort();
}

// Internal helper for making arrays
static void* i9_mkArray(int bytes, int cells) {
    int* memory = I9(_alloc_i)(bytes + 4);
    memory[0] = cells;
    return memory + 1;
}

// Helper: C string to I9 string
static int* i9_mkString(const char* in) {
    int  c;
    int  len = strlen(in);
    int* out = i9_mkArray(len*4, len);

    for (c = 0; c < len; ++c)
        out[c] = (unsigned char)in[c];
    return out;
}

extern void I9(main_paai)(int**);

int main(int argc, char** argv) {
    int c;
    int** args;

    // GC setup. We do point in the "middle" of arrays.
    GC_all_interior_pointers = 1;

    // Create arguments array.
    args = i9_mkArray(sizeof(int*) * argc, argc);
    for (c = 0; c < argc; ++c)
        args[c] = i9_mkString(argv[c]);

    // transfer to program's main
    I9(main_paai)(args);
    return 0;
}


/**
 I/O module
*/

void I9(print_pai) (int* str) {
    int c;
    int len = str[-1];
    for (c = 0; c < len; ++c)
        fputc(str[c], stdout);
}

void I9(println_pai) (int* str) {
    I9(print_pai)(str);
    fputc('\n', stdout);
}

int* I9(readln_ai) (void) {
    char* line = i9util_readLine();
    int*  str  = i9_mkString(line);
    free(line);
    return str;
}

int I9(getchar_i) (void) {
    // ### behavior on eof is unspecified
    return fgetc(stdin);
}

int I9(eof_b) (void) {
    return feof(stdin) ? 1 : 0;
}

/**
 Conv module
*/

typedef struct convResult {
    int num;
    int ok; // must write the entire into to ensure full width is set
} convResult;

void I9(parseInt_t2ibai) (convResult* retOut, int* str) {
    // ### should this worry about overflow?
    int c;
    int len = str[-1];
    int neg = 0;

    retOut->num = 0;
    retOut->ok  = 0;

    if (!len)
        return;

    if (str[0] == '-')
        neg = 1;

    for (c = neg; c < len; ++c) {
        if (str[c] >= '0' && str[c] <= '9') {
            retOut->num = 10*retOut->num + (str[c] - '0');
        } else {
            retOut->num = 0;
            return; // returning (0, false);
        }
    }

    retOut->ok = 1;
    if (neg)
        retOut->num = -retOut->num;
}

int* I9(unparseInt_aii) (int in) {
    char buf[16]; // more than enough to represent numbers.
    sprintf(buf, "%d", in);

    return i9_mkString(buf);
}

/**
 Assert module
*/
void I9(assert_pb) (int cond) {
    if (!cond) {
        fprintf(stderr, "Assertion failed\n");
        abort();
    }
}

/**
 Timer module. This ab(uses) the int[] type for
 GC'd opaque objects.
*/
struct timeval* I9(getTimestamp_ai) (void) {
    struct timeval* tStamp = i9_mkArray(sizeof(struct timeval), 0);
    gettimeofday(tStamp, 0);
    return tStamp;
}

int I9(timestampDifference_iaiai)(int* l, int* r) {
    struct timeval* lTime = (struct timeval*)l;
    struct timeval* rTime = (struct timeval*)r;

    int secondsDiff = lTime->tv_sec  - rTime->tv_sec;
    int usecDiff  = lTime->tv_usec - rTime->tv_usec; // micro is 1e6, so the range is fine.

    long combinedDiff   = usecDiff + secondsDiff * 1000000L;
    return (int)(combinedDiff/1000);
}


// kate: indent-width 4; replace-tabs on; tab-width 4; space-indent on;
// vim: sw=4 ts=4 et
