#ifndef IX_QOBJECT_HOLDER_H
#define IX_QOBJECT_HOLDER_H

#include <cassert>
#include <QObject>
#include <QPointer>

#include <ixobj.h>
#include <gcpin.h>

// Since Qt methods will return QObjects and not the wrappers,
// we keep a table that lets us do reverse lookup.
IXobj* ixqGetWrapper(QObject* object);
void   ixqPutWrapper(QObject* object, IXobj* wrapper);
void   ixqForgetWrapper(QObject* object);

// This is a helper to manage pinning/unpinning/deletion of QObjects
// wrapped by IX objects. The basic invariant is that QObjects
// with parents are managed by C++ and are pinned; those without
// parents are managed by the IX GC.
//
// Because of pinning, this should be included by value in the
// wrapper object
template<typename T> class IXQObjectHolder
{
private:
    QPointer<T> handle; // the use of QPointer here isn't needed for
                        // operation, but is rather a precaution
                        // to make crashes more predictable in case of
                        // mistakes

    GCPin pin;
public:
    IXQObjectHolder(): pin(this)
    {}

    T* get() {
        return handle.data();
    }

    bool isNull() const {
        return handle.isNull();
    }

    void setObject(T* object, IXobj* wrapper) {
        assert (handle.isNull());
        handle = object;
        if (object->parent())
            pin.pin();
        ixqPutWrapper(object, wrapper);
    }

    void parentChange() {
        if (handle->parent())
            pin.pin();
        else
            pin.unpin();
    }

    ~IXQObjectHolder() {
        // if we are not pinned, we own the object, so we should delete it.
        // (and if we're pinned, we should not be here!)
        assert (!pin.isPinned());

        ixqForgetWrapper(handle.data());
        delete handle.data();
    }
};

#endif
// kate: indent-width 4; replace-tabs on; tab-width 4; space-indent on;
