#!/usr/bin/env ruby
# Benchmark driver
# Stuff below is configuration.
# The script's expectations are as follows:
#   it looks for test files in the bench/ subdirectory,
#   and for src.jar files in all subdirs one level below...
#   or you can write something like $jars = [ "myCompiler.jar" ] at around
#   line 206. It will produce a file name benchlog-something.txt,
#   with something based on directory name, as well as matrix.html.
#   Put that together with matrix.js to view.


BaseClassPath = "/home/maksim/jars/cs4120.jar:/home/maksim/jars/java-cup-11a-runtime.jar:" +
                "/home/maksim/jars/args4j-2.0.10.jar:/home/maksim/jars/commons-lang-2.4.jar"
LinkScript    = "/home/maksim/Academic/CS/TA/412-2/2009fa/hw/pa5/runtime/linki9.sh"

Exclude = [ "bench/io.i9", "bench/assert.i9", "bench/conv.i9",
            "bench/timer.i9", "bench/mkmatrix.i9"]

RUNS = 5;

require 'timeout'
require 'find'
require 'benchmark'

$pass = 0
$fail = 0

Red    = "\x1b[1;31;40m"
Green  = "\x1b[0;32;40m"
Normal = "\x1b[0;37;40m"

def fail(f)
    print Red
    print "\t[FAIL]"
    puts Normal
    f.puts("FAIL")
    f.puts("="*80)
    $fail = $fail + 1
    $stdout.flush
end

def pass(f)
    print Green
    print "\t[PASS]"
    puts Normal
    f.puts("PASS")
    f.puts("="*80)
    $pass = $pass + 1
    $stdout.flush
end

def jarMain(jarFile)
    cmd = "unzip -C -p #{jarFile} META-INF/MANIFEST.MF"
    pipe = IO.popen(cmd)
    pipe.readlines.each do |line|
        line.chop!
        m = /Main-Class: (.*)/.match(line)
        if (m) then
            return m[1]
        end
    end
    return nil;
end

def filterRun(prefix, f, cmd)
    begin
        Timeout::timeout(120) do
            IO.popen("./timelimit.sh " + cmd) { |pipe|
                out = pipe.readlines
                out.each do |line|
                    f.puts prefix + line
                end
            }
            return $?.exitstatus
        end
    rescue Timeout::Error
        f.puts "Timeout."
    end
    return 40
end

def dumpFile(f, name, prefix)
    File.open(name, "r") do |inp|
        inp.each_line do |line|
            f.puts prefix + line
        end
    end
end

def rm(fileName)
    begin
        File.unlink fileName
    rescue
    end
end

# Handles compile/link failure tests, and returns true if we're done
def catchRunFailure(f, result, expect, failureExpect, testFile)
    failed     = (result != 0) || (not (File.exists? testFile));
    shouldFail = (expect == failureExpect);
    if (failed != shouldFail) then
        fail(f);
        $html.print ("<td>BuildFAIL</td><td></td>")
        return true
    end

    # if we were expected to fail and did, the test passed.
    if (shouldFail) then
        pass(f)
        $html.print ("<td>BuildFAIL</td><td></td>")
        return true
    end

    return false
end

def runTest(f, html, jar, testFileName, opts)
    # dumpFile(f, testFileName, "<II>");

    expect = "ok"

    mainClass = jarMain(jar);

    if not mainClass then
        puts "Unable to determine main class for jar!"
        return
    end

    sFileName = testFileName.sub(".i9", ".s");
    refFile   = testFileName.sub(".i9", ".ref");

    f.puts "Compiling";
    rm sFileName
    cmd = "java -cp #{BaseClassPath}:#{jar} #{mainClass} #{opts} #{testFileName} 2>&1"

    result = nil;
    time =  Benchmark.realtime {
        result = filterRun("<IC>", f, cmd);
    }
    f.puts time
    print " " + time.to_s + " s ";

    return if catchRunFailure(f, result, expect, "compileFail", sFileName)

    f.puts "Linking";
    rm "/tmp/studentBin";
    rm "/tmp/studentOut"
    cmd = "#{LinkScript} #{sFileName} bench/mkmatrix.c -o /tmp/studentBin";
    result = filterRun("<IL>", f, cmd);
    return if catchRunFailure(f, result, expect, "linkFail", "/tmp/studentBin")

    html.print("<td>#{time.to_s}</td>");

    f.puts "Running";

    # provide input for the one test that needs it
    extra = "";
    extra = "< #{testFileName}" if (testFileName.end_with? "lex-bench.i9");

    alreadyFailed = false;
    html.print("<td>");
    RUNS.times {
        if not alreadyFailed then
            time = Benchmark.realtime {
                result = filterRun("<PO>", f, "/tmp/studentBin #{extra} > /tmp/studentOut  2>&1");
            }
        end

        if (result == 0) then
            html.print(time.to_s + ",")
        else
            html.print("RunFAIL,")
            alreadyFailed = true
        end
        html.flush
    }
    html.print("</td>")

    # Compare output...
    filterRun("<DF>", f, "diff --text -u #{refFile} /tmp/studentOut ");

    # Figure out the result code we want.
    if (result == 0) then
        pass(f);
    else
        f.puts("Got code:#{result.to_s}!")
        fail(f);
    end
end

$testfiles = [];

Find.find("bench/") do |test|
    puts test
    next if not test.end_with? ".i9"
    next if Exclude.include? test
    $testfiles << test
end

$testfiles.sort!

$jars = []

Dir.glob("**/src.jar").each do |jar|
    $jars << jar
end

puts "$jars = [ " + $jars.join(",\n\t") + "\n]";

$html = File.new("matrix.html", "w");
$html.puts('<script src="matrix.js"></script>');
$html.puts('<body onload="prettifyTable()">');
$html.puts('<table id="t" border=1 cellspacing=0><thead>');
$html.print "<tr>";
$html.print "<th rowspan=2>Jar</th>";

$testfiles.each do |test|
    test = test.sub("bench/", "")
    test = test.sub(".i9", "")
    $html.print("<th colspan=4>#{test}</th>");
end
$html.puts("</tr>")
$html.print("<tr>")
$testfiles.each do |test|
    $html.print("<td>Comp +O</td>")
    $html.print("<td>Run +O</td>")
    $html.print("<td>Comp -O</td>")
    $html.print("<td>Run -O</td>")
end
$html.puts "</tr>";
$html.puts "<tbody>";

$jars.each do |jar|
    puts("Testing: " + jar)
    $html.print("<tr><td>" + jar + "</td>");
    group   = jar.sub("/src.jar", "")
    logName = jar.sub("src.jar", "benchlog-#{group}.txt")
    File.open(logName, "w") do |f|
        $pass = 0
        $fail = 0
        $testfiles.each do |test|
            print("\t" + test);
            f.puts("Testing:" + test);
            $html.print("<!-- #{test} -->");
            runTest(f, $html, jar, test, "");
            print("\t" + test + " (-O)");
            f.puts("Testing:" + test + " (-O)");
            runTest(f,  $html, jar, test, "-O");
        end
        f.puts "(-O) PASS:#{$pass.to_s} FAIL:#{$fail.to_s}"
        puts "(-O) PASS:#{$pass.to_s} FAIL:#{$fail.to_s}"
    end
    $html.puts "</tr>"
end
