#include "arr.h"
#include <stdio.h>
#include <string.h>

// An implementation of Turing's "bombe" attack on the Enigma cipher.
// Essentially, given a crib, it's possible to figure out the position of the
// rotors independently from plugboard (which has far more settings); this does
// involve bruteforce, by 26^3 is doable even with electromechanical means.
//
// (With rotors known, the plugboard can be reverse-engineered by hand...)
//
// This was originally a student solution to a homework
// given in CSC290, Fall 2002 @ University of Rochester
//
// The crib from which the loops were extracted as:
//             1         2        3          4         5
//   0123456789012345678901234567890123456789012345678901234567
//   theenigmacipheriscertainlystrongbutnotwithoutitsweaknesses
//   JCJYROSNHZXNTNGQPXHADDUSNKCGWZEOQFVLBDLFFDZQLJSHYMFNHWLYXY

int toLower(int c) {
    if ((c >= 'A') & (c <= 'Z'))
        return c - 'A' + 'a';
    return c;
}

// Helper: C string to I9-like array
static int* strArray(const char* in) {
    int  c;
    int  len = strlen(in);
    int* out = arr_alloc(len);

    for (c = 0; c < len; ++c)
        out[c] = (unsigned char)in[c];
    return out;
}

///////////////////////////////////////////////////////////////////////////////
/// Rotor

void makeInverse(int* base, int* inv) {
    int c = 0;
    while (c < 26) {
        //We have c->base[c], inv has base[c] going to c.
        arr_at(inv, arr_at(base, c)) = c;
        c = c + 1;
    }
}

int** mkMatrix(int size) {
    int c;
    int** topLevel = (int**)arr_alloc(size);
    for (c = 0; c < size; ++c)
        topLevel[c] = arr_alloc(size);
        // no check here since the other mkmatrix is in C as well.
    return topLevel;
}

typedef struct
{
    int** forward;
    int** backward;
    int   pos;
} Rotor;

// Just a version of arrays-of-arrays
#define arr2_at(a, idx) (int*)arr_at((int*)a, idx)

// Rotor encoding: forward map, reverse map, position
Rotor makeRotor(int* sig) {
    Rotor r;
    r.forward  = mkMatrix(26);
    r.backward = mkMatrix(26);
    r.pos      = 0;

    int* base = arr_alloc(26);

    int c = 0;
    while (c < 26) {
        int cd = toLower(arr_at(sig, c)) - 'a';
        arr_at(base, c) = cd;
        c = c + 1;
    }

    int* inv = arr_alloc(26);

    int rot = 0;
    while (rot < 26) {
        //Fill in forward direction for this..
        c = 0;
        while (c < 26) {
            arr_at(arr2_at(r.forward, rot), c) = arr_at(base, c);
            c = c + 1;
        }

        //Make inverse..
        makeInverse(base, inv);

        //Fill in backward
        c = 0;
        while (c < 26) {
            arr_at(arr2_at(r.backward, rot), c) = arr_at(inv, c);
            c = c + 1;
        }

        //Simulate rotation..
        int first = (arr_at(base, 0) - 1 + 26)%26;
        int pos = 1;
        while (pos < 26) {
            arr_at(base, pos-1) = (arr_at(base, pos) - 1 + 26)%26;
            pos = pos + 1;
        }

        arr_at(base, 25) = first;

        rot = rot + 1;
    }

    return r;
}

int rotorEncryptForward(Rotor r, int letter) {
    return arr_at(arr2_at(r.forward, r.pos), letter);
}

int rotorEncryptBack(Rotor r, int letter) {
    return arr_at(arr2_at(r.backward, r.pos), letter);
}

Rotor rotorSetPosition(Rotor r, int pos) {
    r.pos = pos;
    return r;
}

///////////////////////////////////////////////////////////////////////////////
/// Reflector

// This is just a permutation, so its state is an int*

int* makeReflector(int* encoding) {
    int* perm = arr_alloc(26);

    //Extract the base permutation.
    int c = 0;
    while (c < 26) {
        int cd = toLower(arr_at(encoding, c))-'a';
        arr_at(perm, c) = cd;
        c = c + 1;
    }
    return perm;
}

int reflectorEncrypt(int* perm, int l) {
    return arr_at(perm, l);
}

int main() {
    int** loops = (int**)arr_alloc(4);

    int loop1[9] = {8, 12, 27, 6, 57, 25, 51, 52, -1};
    int loop2[9] = {8, 12, 27, 6, 55, 25, 51, 52, -1};
    int loop3[5] = {4, 12, 46, 47, -1};
    int loop4[8] = {7, 12, 27, 6, 16, 11, 52, -1};

    loops[0] = loop1 + 1;
    loops[1] = loop2 + 1;
    loops[2] = loop3 + 1;
    loops[3] = loop4 + 1;

    //Setup components
    Rotor r1 = makeRotor(strArray("EKMFLGDQVZNTOWYHXUSPAIBRCJ"));
    Rotor r2 = makeRotor(strArray("AJDKSIRUXBLHWTMCQGZNPYFVOE"));
    Rotor r3 = makeRotor(strArray("BDFHJLCPRTXVZNYEIWGAKMUSQO"));
    int*  mb = makeReflector(strArray("YRUHQSLDPXNGOKMIEBFZCWVJAT"));

    int pos = 0;
    while (pos<26*26*26) {
        //Guess where the first letter in the loop goes to..
        int guess = 0;
        while (guess < 26) {
            int allMatch = 1;
            int loop = 0;
            while (loop < arr_len((int*)loops)) {
                int l = guess;
                //Try to go through the possible values..
                int loopPos = 0;
                while (arr_at(arr2_at(loops, loop), loopPos) != -1) {
                    int epos = pos + arr_at(arr2_at(loops, loop), loopPos);

                    r1 = rotorSetPosition(r1, epos % 26);
                    r2 = rotorSetPosition(r2, (epos/26)%26);
                    r3 = rotorSetPosition(r3, (epos/(26*26)%26));
                    l = rotorEncryptForward(r1, l);
                    l = rotorEncryptForward(r2, l);
                    l = rotorEncryptForward(r3, l);
                    l = reflectorEncrypt(mb, l);
                    l = rotorEncryptBack(r3, l);
                    l = rotorEncryptBack(r2, l);
                    l = rotorEncryptBack(r1, l);
                    loopPos = loopPos + 1;
                }

                if (l != guess)
                    allMatch = 0;

                loop = loop + 1;
            } // while loop

            if (allMatch) {
                char posStr[3];
                posStr[0] = (pos%26)+'A';
                posStr[1] = ((pos/26)%26)+'A';
                posStr[2] = ((pos/(26*26))%26)+'A';
                printf ("MATCH At rotor pos:%c%c%c",
                        posStr[0], posStr[1], posStr[2]);
                printf (" first comes in from: %c\n", guess + 'A');
            }

            guess = guess + 1;
        } // while guess

        pos = pos + 1;
    } // while pos
    return 0;
}
