#include <stdio.h>
#include <string.h>
#include "system.h"
#include "io.h"
#include "linearalg.h"
#include "GLout.h"

#define InitWindowWidth	400
#define InitWindowHeight	300

int RunRad(int argc, char **argv);

int RunRad(int argc, char **argv) {

  char commfilename[256];/*Array to hold the name of the commfile */
  /*parameters */
  int m;
  double t,L, dt, T, dx;
  
  char Cfilename[256]; /* holds name of file with initial C */
  double *C; /* The * indicates that C is a "pointer"--it will hold a
                memory address rather than a value.  In this case, the
                address will be the start of an array that will be
                created below. */
  double *RHS;/*b, the "right-hand side of matrix probleim*/
  int *iq,*jq;/* iq & jq define the position of the non-zero values in A */
  double *A;  /* the values of the matrix */
                
  /*indicate cases of the data, their names, their values, their times,
    the number of times, and their current iteration */
  int kcase, ucase, gcase, Kcase; 
  char kfilename[256], ufilename[256], gfilename[256], Kfilename[256];
  double **k, **u, **g, **K; /* values held in 2D arrays (hence **) */
  double *ktimes, *utimes, *gtimes, *Ktimes; /* 1D arrays */
  int kN, uN, gN, KN;  
  int kJ, uJ, gJ, KJ;  
  
  double *sigma, *lambda, *kvec, *uvec, *gvec, *Kvec;
  
  int i, j; /* iterators */
  int left, right, NSamp, SampNum;
  double NextSamp;
  
  printf("\n\n          Welcome to RAD1D\n\n");
  printf("     1D Reaction-Advection-Diffusion\n\n");
  
            /********************************************/
            /*                                          */
            /*            Get Parameters                */
            /*                                          */
            /********************************************/
  
  /*   First, get the name of the input file  */
  /*   from the user or command line option   */
  
  if(argc > 1) { /* arguments were passed */
    strcpy(commfilename, argv[1]);
    if(argc >2){
    	sscanf(argv[2],"%d",&NSamp);
    }else{
    	printf("Enter the number of time samples:  ");
    	scanf("%d", &NSamp);
    }
    
  } else { /* no arguments, need to get comm file from user */
    printf("Enter the name of the command file:  ");
    scanf("%s", &commfilename[0]);
        /*SCANF parses the users input as a string (%s) and
          places results in commfilename*/
    printf("Enter the number of time samples:  ");
    scanf("%d", &NSamp); 
  }
  
  /*   Read the parameters and file names from */
  /*   commfilename. Ensures that kcase, ucase */
  /*   etc. are 0,1, or 2                      */
  
  ReadComm(commfilename, &m, &L, &dt, &T, Cfilename,
            &ucase, ufilename, &kcase, kfilename,  
            &gcase, gfilename, &Kcase, Kfilename ); 
            
  dx=L/(m-1.0); /* derived parameter */
  ReportParams(m,L,dx,dt,T);  /* Print paramaters--allows user to check  */
  
  
            /********************************************/
            /*                                          */
            /*                Get Data                  */
            /*                                          */
            /********************************************/
  C=newarray_double(m, "main", "Unable to create C (length m)");
    
  
  printf("Reading initial concentrations from %s\n",Cfilename);
  GetInitialCond(Cfilename, C, m); /* get initial C */
  printf("%s read successfully\n",Cfilename);
  
    SampNum=1;
    NextSamp=T*SampNum/(NSamp-1.0);

  InitGLArrays(C,m,NSamp,L,dx);/* Allocates space for GLC and GLx (in GLout.c) */
  
    /********************************************/
    /*         Call GetData for each term       */
    /********************************************/
    GetData(ucase, ufilename, &u, &utimes, 
                m, &uN, T, "advection", "u");
    uJ=0;

    GetData(kcase, kfilename, &k, &ktimes, 
                m, &kN, T, "diffusion", "k");
    kJ=0;


    GetData(gcase, gfilename, &g, &gtimes, 
                m, &gN, T, "reaction", "g");
    gJ=0;


    GetData(Kcase, Kfilename, &K, &Ktimes, 
                m, &KN, T, "reaction", "K");
    KJ=0;
 
  
            /********************************************/
            /*                                          */
            /*      Initialize Variables                */
            /*                                          */
            /********************************************/
    /* We still need to allocate storage for RHS (b) and A */
    
    RHS=newarray_double(m, "main", "Unable to create RHS (length m)");
    
    iq=newarray_int(m,"main", "Unable to create IQ (length m)");
    jq=newarray_int(m*3,"main","Unable to create JQ (length m*3)");
    A=newarray_double(m*3,"main","Unable to create A (length m*3)");
    i=0;
    for(j=0;j<m;j++){
        iq[j]=j*3+2;
        
        if(j>0){
            jq[i]=j-1;
        }else{
            jq[i]=m-1; /* connect node 0 to node m-1 */
        }
        jq[i+1]=j;   
        if(j<(m-1)){
            jq[i+2]=j+1;
        }else{
            jq[i+2]=0;/* connect node m-1 to node 0 */
        }
        i=i+3;
    }
      
    t=0.0;
    
    /*  Create vectors uvec, kvec, gvec, and Kvec to store */
    /*  current values and update them for time t=0        */
    
    kvec =newarray_double(m,"main","Unable to create kvec (length m)");
    uvec=newarray_double(m,"main","Unable to create uvec (length m)");
    gvec =newarray_double(m,"main","Unable to create gvec (length m)");
    Kvec =newarray_double(m,"main","Unable to create Kvec (length m)");
    
    sigma =newarray_double(m,"main","Unable to create sigma (length m)");
    lambda=newarray_double(m,"main","Unable to create lambda (length m)");
    
    if(kcase==2){
    	GetCurrentValue(t,ktimes,&kJ,k,kN,m,kvec, "k");
    } else {
        for(j=0;j<m;j++){
            kvec[j]=k[0][j];
        }
    }
    if(ucase==2){
    	GetCurrentValue(t,utimes,&uJ,u,uN,m,uvec, "u");
    } else {
        for(j=0;j<m;j++){
            uvec[j]=u[0][j];
        }
    }
    
    if(gcase==2){
    	GetCurrentValue(t,gtimes,&gJ,g,gN,m,gvec, "g");
    } else {
        for(j=0;j<m;j++){
            gvec[j]=g[0][j];/* lambda=u*dt */
        }
    }
    if(Kcase==2){
    	GetCurrentValue(t,Ktimes,&KJ,K,KN,m,Kvec, "K");
    } else {
        for(j=0;j<m;j++){
            Kvec[j]=K[0][j];/* lambda=u*dt */
        }
    }
    

    while(t<T){
        if(t>NextSamp){
            SampNum=SampNum+1;
            NextSamp=T*SampNum/(NSamp-1.0);
            AddSample(C, t);
        }
        for(j=0;j<m;j++){
            lambda[j]=uvec[j];
        }
        
        /* add sigma to lambda if kcase >0 */
        if(kcase > 0){
            for(j=0;j<m;j++){
                left=j-1;  /*node to the left */
                right=j+1; /*node to the right */
                if(left<0){
                    left=m-1;
                }
                if(right>m-1){
                    right=0;
                }
                lambda[j]=lambda[j]+0.5*(kvec[right]-kvec[left])/dx;/* lambda=u*dt */
            }
        }
        
        for(j=0;j<m;j++){
            lambda[j]=lambda[j]*dt/dx;
            sigma[j]=kvec[j]*dt/dx/dx;
        }
        
        if(t<dt || kcase==2){
            BuildA(A,m,sigma);
        }
        for(j=0;j<m;j++){
            left=j-1;  /*node to the left */
            right=j+1; /*node to the right */
            if(left<0){
                left=m-1;
            }
            if(right>m-1){
                right=0;
            }
            RHS[j]=C[j]+0.5*lambda[j]*(C[right]-C[left]);
            RHS[j]=RHS[j]+0.5*lambda[j]*lambda[j]*(C[right]-2*C[j]+C[left]);
            RHS[j]=RHS[j]+dt*gvec[j]*(Kvec[j]-C[j])/Kvec[j];
        }
        
        SolveA(iq,jq,A,RHS,C,m);
        
        t=t+dt;
        if(t<T){
            if(kcase==2){
                GetCurrentValue(t,ktimes,&kJ,k,kN,m,kvec,"k");
            }
            if(ucase==2){
                GetCurrentValue(t,utimes,&uJ,u,uN,m,uvec,"u");
            }
            if(gcase==2){
                GetCurrentValue(t,gtimes,&gJ,g,gN,m,gvec,"g");
            }
            if(Kcase==2){
                GetCurrentValue(t,Ktimes,&KJ,K,KN,m,Kvec,"K");
            }
        }
    }
        if(SampNum<=NSamp){
            AddSample(C, t);
        }
  
    PrintArray("Cfinal.txt", C, m);  /* Saves C in a file called C.txt */
  
    return 0;
}

int main(int argc, char** argv)
{
	int j;
        double L=10.0;
        double dx,xj,pi;
        
        glutInit(&argc, argv);
	glutInitDisplayMode (GLUT_RGB | GLUT_DEPTH);
	glutInitWindowSize (InitWindowWidth, InitWindowHeight);
	glutInitWindowPosition (100, 100);
	glutCreateWindow ("RAD1D-1D Reaction-Advection-Diffusion");
        glutDisplayFunc(DrawC);
        glutReshapeFunc(ResizeIt);
        glutSpecialFunc(EmphasizeTime);
        /* Set-up GL */
	InitGL(InitWindowWidth, InitWindowHeight);
        /* Run RAD1D */
        printf("Starting RunRad\n");
        RunRad(argc, argv);
        /* Turn control to GL */
        printf("Starting GL main\n");
	glutMainLoop();

	return 0;
}


