/*
 *  linearalg.c
 *  RAD1D
 *
 *  Created by Andrew Pershing on Tue Mar 05 2002.
 *
 */

#include "linearalg.h"
#include "system.h"

/* Prototypes for local functions */
void sprsmlt(double *qv, int *iq, int *jq, double *x, double *b, int nn);
int pcgm(double *a, double *x, int nn, int *iq, int *jq);


void GetCurrentValue(double t, double *times,int *J, double **V, int N, int m, double *vec, char varname[]){
    /* Gets current value of V (N-by-m) where each row[j] in V was observed at time=times[j] */
    /* First, this routine finds J s.t. times[J-1]<=t<times[J]. It then computes the value   */
    /* of V at each of the m grid points by linear interpolation.                            */

    char msg[256];
    int j;
    while(times[*J]<t & *J<N){
        *J=*J+1;
    }
    if(*J>=N){
        sprintf(msg, "vector of times for %s does not encompass t=%f (max time=%f)",varname,t,times[N-1]);
        Error("GetCurrentValue",msg);
    }
    for(j=0;j<m;j++){
        vec[j]=(t-times[*J])*(V[*J][j]-V[*J-1][j])/(times[*J]-times[*J-1]);
    }
}

void BuildA(double A[], int m, double sigma[]){
    /* Builds A using values in sigma  */
    int i, j;
    int left, right;
    
    i=0;
    for(j=0;j<m;j++){
        left=j-1;  /*node to the left */
        right=j+1; /*node to the right */
        if(left<0){
            left=m-1;
        }
        if(right>m-1){
            right=0;
        }
        
        A[i]= -1.0*sigma[left];
        A[i+1]= 1.0+2.0*sigma[j];
        A[i+2]= -1.0*sigma[right];
        i=i+3;
    }

}


void SolveA(int iq[], int jq[], double A[], double RHS[], double C[], int m){
    /* Solves A*C=RHS where A is m-by-m (described in SPRSPAK */
    /* form using IQ and JQ), and RHS and C are length m      */
    int j;
    
    pcgm(A, RHS, m, iq, jq);/*pre-conditioned conjugate gradient solver, RHS is overwritten with answer */
    for(j=0;j<m;j++){
        C[j]=RHS[j]; /* copy answer into C */
    }
}

void sprsmlt(double *qv, int *iq, int *jq, double *x, double *b, int nn){
/* ----------------------------------------------------------------------- */
/*   MULTIPLIES QV*X; ANSWER RETURNS IN B                                  */
/*   NN IS THE SIZE OF Q (# OF EQUATIONS)                                  */
/*                                                                         */
/*   Notes: This version differs from the SPRSPAK version in 2 ways:       */
/*            1.(*)'s are used to dimension agruments [instead of [0]'s],  */
/*            2.The do loops have been modified to a DO-ENDDO style.       */
/*          Modified by Christopher E. Naimie                              */
/* ----------------------------------------------------------------------- */
  int i, k, kmin, kmax;
  double sum;
  kmin=0;
  for(i=0; i<nn; i++){
    sum=0.0;
    kmax=iq[i];
    for(k=kmin; k<=kmax; k++){
      sum=sum+qv[k]*x[jq[k]];
    } /*  for  */
    b[i]=sum;
    kmin=kmax+1;
  } /*  for  */
} /* routine */



int pcgm(double *a, double *x, int nn, int *iq, int *jq){
/* -----+----------------------------------------------------------------+ */
/*  Solve Ax=b by the preconditioned conjugate gradient method, where      */
/*  A is a sparse, symmetric and positiv definite matrix, b a known        */
/*  vector, and x the unknown vector. In the call b is x.                  */
/*  The preconditioner C is a diagonal matrix stored in a vector.          */
/*                                                                         */
/*  Nicolai Kliem, 11 Feb. 1998                                            */
/* -----+----------------------------------------------------------------+ */
/*                                                                         */
/*  Local NN-arrays                                                        */
/*                                                                         */

  static double *p, *r, *w, *b, *z, *c;
  static int trip=0;
/*                                                                         */
/*  Local varialbes and parameters                                         */
/*                                                                         */
  double  alpha,beta,rz,rznew,rzstop,pw,rtmp,epsilon=1.e-12;

  int k,n,kmax;
  
  char *sub, *msg;

/*                                                                         */
/*  Create local arrays                                                    */
/*                                                                         */
  sub="fixsubs->PCGM";
  msg="Unable to create local arrays";
  
  if(0==trip){
    p=newarray_double(nn, sub, msg);
    r=newarray_double(nn, sub, msg);
    w=newarray_double(nn, sub, msg);
    b=newarray_double(nn, sub, msg);
    z=newarray_double(nn, sub, msg);
    c=newarray_double(nn, sub, msg);
  }
  trip+=1;
  

/* -----+----------------------------------------------------------------+ */
/*  Initialize                                                             */
/*  x=0                                                                    */
/* -----+----------------------------------------------------------------+ */

/*  The preconditioner C could be calculated once and forever in           */
/*  subroutine STATIONARYQ4, but this recuires that the C is transfered    */
/*  through the subroutine calls, and is not implemented here to make as   */
/*  few changes as possible.                                               */
/*                                                                         */
/*  Only returns local nodes of X                                          */
/*                                                                         */
  for(n=0; n<nn; n++){
    k=3*n+1;/* Assumes A is tridiagonal kay(n,n,iq,jq); */
    rtmp=a[k];
    if(rtmp == 0.0) {
      Error(sub, "Divide by zero");
    } 
    c[n]=1./rtmp;
  } 

  for(n=0; n<nn; n++){
    r[n]=x[n];
    b[n]=x[n];
    x[n]=0.0;
    w[n]=0.0;
  } 

  for(n=0; n<nn; n++){
    z[n]=c[n]*r[n];
    p[n]=z[n];
  } 

  rz=0.0;
  for(n=0; n<nn; n++){
    rz=rz+r[n]*z[n];
  } 
  kmax=nn;
  k=0;
  rzstop=epsilon*rz;
  
/* -----+----------------------------------------------------------------+ */
/*  Loop                                                                   */
/* -----+----------------------------------------------------------------+ */

  while ((k < kmax) && (rz > rzstop)){
/*                                                                         */
/*  w=A*p                                                                  */
/*                                                                         */
/*  compute A*p=w                                                          */
    sprsmlt(a,iq,jq,p,w,nn);
/*                                                                         */
/*  alpha=rz/(pT*w)                                                        */
/*                                                                         */
    pw=0.0;
    for(n=0; n<nn; n++){
      pw=pw+p[n]*w[n];
    } 
    alpha=rz/pw;
/*                                                                         */
/*  x=x+alpha*p                                                            */
/*                                                                         */
    for(n=0; n<nn; n++){
      x[n]=x[n]+alpha*p[n];
    } 
/*                                                                         */
/*  r=r-alpha*w                                                            */
/*                                                                         */
    for(n=0; n<nn; n++){
      r[n]=r[n]-alpha*w[n];
    } 
/*                                                                         */
/*  z=C-1*r                                                                */
/*                                                                         */
    for(n=0; n<nn; n++){
      z[n]=c[n]*r[n];
    } 
/*                                                                         */
/*  rznew=r*z                                                              */
/*  beta=rznew/rz                                                          */
/*  rz=rznew                                                               */
/*                                                                         */
    rznew=0.0;
    for(n=0; n<nn; n++){
      rznew=rznew+r[n]*z[n];
    } 

    beta=rznew/rz;
    rz=rznew;
/*                                                                         */
/*  p=z+beta*p                                                             */
/*                                                                         */
    for(n=0; n<nn; n++){
      p[n]=z[n]+beta*p[n];
    } 
    k=k+1;
  } /*  while  */
  return(k); /* return the number of iterations needed */
} /* routine */



    

