/* Copyright (c) 2006, 2009, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */
 
package com.cburch.logisim.tools;

import java.awt.Cursor;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;

import javax.swing.Icon;

import com.cburch.logisim.circuit.CircuitActions;
import com.cburch.logisim.circuit.Wire;
import com.cburch.logisim.comp.Component;
import com.cburch.logisim.comp.ComponentDrawContext;
import com.cburch.logisim.data.Location;
import com.cburch.logisim.gui.main.Canvas;
import com.cburch.logisim.proj.Action;
import com.cburch.logisim.proj.LogisimPreferences;
import com.cburch.logisim.util.GraphicsUtil;
import com.cburch.logisim.util.Icons;

import java.util.ArrayList;
import java.util.Iterator;

public class WiringTool extends Tool {
    private static Cursor cursor
        = Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR);
    private static final Icon toolIcon = Icons.getIcon("wiring.gif");
    
    private static final int HORIZONTAL = 1;
    private static final int VERTICAL = 2;

    private boolean exists = false;
    private boolean inCanvas = false;
    private Location start = Location.create(0, 0);
    private Location cur = Location.create(0, 0);
    private boolean startShortening = false;
    private boolean shortening = false;
    private Action lastAction = null;
    private int direction = 0;

    public WiringTool() { }
    
    public boolean equals(Object other) {
        return other instanceof WiringTool;
    }
    
    public int hashCode() {
        return WiringTool.class.hashCode();
    }

    public String getName() {
        return "Wiring Tool";
    }

    public String getDisplayName() {
        return Strings.get("wiringTool");
    }

    public String getDescription() {
        return Strings.get("wiringToolDesc");
    }
    
    private boolean computeMove(int newX, int newY) {
        if(cur.getX() == newX && cur.getY() == newY) return false;
        Location start = this.start;
        if(direction == 0) {
            if(newX != start.getX()) direction = HORIZONTAL;
            else if(newY != start.getY()) direction = VERTICAL;
        } else if(direction == HORIZONTAL && newX == start.getX()) {
            if(newY == start.getY()) direction = 0;
            else direction = VERTICAL;
        } else if(direction == VERTICAL && newY == start.getY()) {
            if(newX == start.getX()) direction = 0;
            else direction = HORIZONTAL;
        }
        return true;
    }

    public void draw(Canvas canvas, ComponentDrawContext context) {
        Graphics g = context.getGraphics();
        if(exists && inCanvas) {
            g.setColor(shortening ? Color.WHITE : Color.BLACK);
            GraphicsUtil.switchToWidth(g, 3);
            int startX = start.getX();
            int startY = start.getY();
            int curX = cur.getX();
            int curY = cur.getY();
            if(direction == HORIZONTAL) {
                if(startX != curX) g.drawLine(startX, startY, curX, startY);
                if(startY != curY) g.drawLine(curX, startY, curX, curY);
            } else if(direction == VERTICAL) {
                if(startY != curY) g.drawLine(startX, startY, startX, curY);
                if(startX != curX) g.drawLine(startX, curY, curX, curY);
            }
        } else if(LogisimPreferences.getShowGhosts() && inCanvas) {
            g.setColor(Color.GRAY);
            g.fillOval(cur.getX() - 2, cur.getY() - 2, 5, 5);
        }
    }

    public void mouseEntered(Canvas canvas, Graphics g,
            MouseEvent e) {
        inCanvas = true;
        canvas.getProject().repaintCanvas();
    }

    public void mouseExited(Canvas canvas, Graphics g,
            MouseEvent e) {
        inCanvas = false;
        canvas.getProject().repaintCanvas();
    }

    public void mouseMoved(Canvas canvas, Graphics g, MouseEvent e) {
        snapToGrid(e);
        inCanvas = true;
        int curX = e.getX();
        int curY = e.getY();
        if(cur.getX() != curX || cur.getY() != curY) {
            cur = Location.create(curX, curY);
        }
        canvas.getProject().repaintCanvas();
    }

    public void mousePressed(Canvas canvas, Graphics g,
            MouseEvent e) {
        if(!canvas.getProject().getLogisimFile().contains(canvas.getCircuit())) {
            exists = false;
            canvas.setErrorMessage(Strings.getter("cannotModifyError"));
            return;
        }

        snapToGrid(e);
        start = Location.create(e.getX(), e.getY());
        cur = start;
        exists = true;
        
        startShortening = !canvas.getCircuit().getWires(start).isEmpty();
        shortening = startShortening;

        super.mousePressed(canvas, g, e);
        canvas.getProject().repaintCanvas();
    }

    static void snapToGrid(MouseEvent e) {
	// prevent from drawing wires into negative coordinates
	Canvas.snapToGrid(e);
	e.translatePoint(e.getX() < 0 ? -e.getX() : 0, e.getY() < 0 ? -e.getY() : 0);
    }

    public void mouseDragged(Canvas canvas, Graphics g,
            MouseEvent e) {
        if(exists && inCanvas) {
            snapToGrid(e);
            int curX = e.getX();
            int curY = e.getY();
            if(!computeMove(curX, curY)) return;
    
            Rectangle rect = new Rectangle(start.getX(), start.getY(), 0, 0);
            rect.add(cur.getX(), cur.getY());
            rect.add(curX, curY);
            rect.grow(3, 3);
    
            cur = Location.create(curX, curY);
            super.mouseDragged(canvas, g, e);
            
            shortening = false;
            if(startShortening) {
                for(Iterator it = canvas.getCircuit().getWires(start).iterator(); it.hasNext(); ) {
                    Wire w = (Wire) it.next();
                    if(w.contains(cur)) { shortening = true; break; }
                }
            }
            if(!shortening) {
                for(Iterator it = canvas.getCircuit().getWires(cur).iterator(); it.hasNext(); ) {
                    Wire w = (Wire) it.next();
                    if(w.contains(start)) { shortening = true; break; }
                }
            }
    
            canvas.repaint(rect);
        }
    }

    public void mouseReleased(Canvas canvas, Graphics g,
            MouseEvent e) {
        if(!exists) return;
        exists = false;
	if(!inCanvas) return;

        snapToGrid(e);
        int curX = e.getX();
        int curY = e.getY();
        if(computeMove(curX, curY)) {
            cur = Location.create(curX, curY);
        }
        if(cur.equals(start)) return;

        super.mouseReleased(canvas, g, e);

        Action act;
        if(cur.getY() == start.getY() || cur.getX() == start.getX()) {
            Wire w = Wire.create(cur, start);
            w = checkForRepairs(canvas, w, w.getEnd0());
            w = checkForRepairs(canvas, w, w.getEnd1());
            if(w.getLength() <= 0) return;
            act = CircuitActions.addComponent(canvas.getCircuit(), w, true);
        } else {
            Location m;
            if(direction == HORIZONTAL) {
                m = Location.create(cur.getX(), start.getY());
            } else {
                m = Location.create(start.getX(), cur.getY());
            }
            Wire w0 = Wire.create(start, m);
            Wire w1 = Wire.create(m, cur);
            w0 = checkForRepairs(canvas, w0, start);
            w1 = checkForRepairs(canvas, w1, cur);
            ArrayList ws = new ArrayList(2);
            if(w0.getLength() > 0) ws.add(w0);
            if(w1.getLength() > 0) ws.add(w1);
            if(ws.size() == 0) return;
            act = CircuitActions.addComponents(canvas.getCircuit(), ws);
        }
        canvas.getProject().doAction(act);
        lastAction = act;
    }
    
    private Wire checkForRepairs(Canvas canvas, Wire w, Location end) {
        if(w.getLength() <= 10) return w; // don't repair a short wire to nothing
        if(!canvas.getCircuit().getNonWires(end).isEmpty()) return w;

        int delta = (end.equals(w.getEnd0()) ? 10 : -10);
        Location cand;
        if(w.isVertical()) {
            cand = Location.create(end.getX(), end.getY() + delta);
        } else {
            cand = Location.create(end.getX() + delta, end.getY());
        }

        for(Iterator it = canvas.getCircuit().getNonWires(cand).iterator();
                it.hasNext(); ) {
            Component comp = (Component) it.next();
            if(comp.getBounds().contains(end)) {
                WireRepair repair = (WireRepair) comp.getFeature(WireRepair.class);
                if(repair != null && repair.shouldRepairWire(new WireRepairData(w, cand))) {
                    w = Wire.create(w.getOtherEnd(end), cand);
                    canvas.repaint(end.getX() - 13, end.getY() - 13, 26, 26);
                    return w;
                }
            }
        }
        return w;
    }
    
    public void keyPressed(Canvas canvas, KeyEvent event) {
        switch(event.getKeyCode()) {
        case KeyEvent.VK_BACK_SPACE:
            if(lastAction != null && canvas.getProject().getLastAction() == lastAction) {
                canvas.getProject().undoAction();
                lastAction = null;
            }
	    break;
	case KeyEvent.VK_ESCAPE:
	    exists = false;
	    canvas.getProject().repaintCanvas();
	    break;
        }
    }

    public void paintIcon(ComponentDrawContext c, int x, int y) {
        Graphics g = c.getGraphics();
        if(toolIcon != null) {
            toolIcon.paintIcon(c.getDestination(), g, x + 2, y + 2);
        } else {
            g.setColor(java.awt.Color.black);
            g.drawLine(x + 3, y + 13, x + 17, y + 7);
            g.fillOval(x + 1, y + 11, 5, 5);
            g.fillOval(x + 15, y + 5, 5, 5);
        }
    }

    public Cursor getCursor() { return cursor; }
}
