/* Copyright (c) 2006, 2009, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */
 
package com.cburch.logisim.tools;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Graphics;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;

import javax.swing.Icon;
import javax.swing.JOptionPane;

import com.cburch.logisim.circuit.Circuit;
import com.cburch.logisim.circuit.CircuitActions;
import com.cburch.logisim.circuit.CircuitException;
import com.cburch.logisim.comp.Component;
import com.cburch.logisim.comp.ComponentFactory;
import com.cburch.logisim.comp.ComponentDrawContext;
import com.cburch.logisim.data.Attribute;
import com.cburch.logisim.data.AttributeEvent;
import com.cburch.logisim.data.AttributeListener;
import com.cburch.logisim.data.AttributeSet;
import com.cburch.logisim.data.Bounds;
import com.cburch.logisim.data.Direction;
import com.cburch.logisim.data.Location;
import com.cburch.logisim.gui.main.Canvas;
import com.cburch.logisim.proj.Action;
import com.cburch.logisim.proj.Dependencies;
import com.cburch.logisim.proj.LogisimPreferences;
import com.cburch.logisim.proj.Project;
import com.cburch.logisim.util.StringUtil;

public class AddTool extends Tool {
    private static int SHOW_NONE    = 0;
    private static int SHOW_GHOST   = 1;
    private static int SHOW_ADD     = 2;
    private static int SHOW_ADD_NO  = 3;

    private static Cursor cursor
        = Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR);

    private class MyAttributeListener implements AttributeListener {
        public void attributeListChanged(AttributeEvent e) {
            bounds = null;
        }
        public void attributeValueChanged(AttributeEvent e) {
            bounds = null;
        }
    }

    private Class descriptionBase;
    private FactoryDescription description;
    private boolean sourceLoadAttempted;
    private ComponentFactory factory;
    private AttributeSet attrs;
    private Bounds bounds;
    private boolean shouldSnap;
    private int lastX = Integer.MAX_VALUE;
    private int lastY = Integer.MAX_VALUE;
    private int state = SHOW_GHOST;
    private Action lastAddition;
    
    public AddTool(Class base, FactoryDescription description) {
        this.descriptionBase = base;
        this.description = description;
        this.sourceLoadAttempted = false;
        this.shouldSnap = true;
        this.attrs = new FactoryAttributes(base, description);
        attrs.addAttributeListener(new MyAttributeListener());
    }

    public AddTool(ComponentFactory source) {
        this.description = null;
        this.sourceLoadAttempted = true;
        this.factory = source;
        this.bounds = null;
        this.attrs = new FactoryAttributes(source);
        attrs.addAttributeListener(new MyAttributeListener());
        Boolean value = (Boolean) source.getFeature(ComponentFactory.SHOULD_SNAP, attrs);
        this.shouldSnap = value == null ? true : value.booleanValue();
    }

    private AddTool(AddTool base) {
        this.descriptionBase = base.descriptionBase;
        this.description = base.description;
        this.sourceLoadAttempted = base.sourceLoadAttempted;
        this.factory = base.factory;
        this.bounds = base.bounds;
        this.shouldSnap = base.shouldSnap;
        this.attrs = (AttributeSet) base.attrs.clone();
        attrs.addAttributeListener(new MyAttributeListener());
    }
    
    public boolean equals(Object other) {
        if(!(other instanceof AddTool)) return false;
        AddTool o = (AddTool) other;
        if(this.description != null) {
            return this.descriptionBase == o.descriptionBase
                && this.description.equals(o.description);
        } else {
            return this.factory.equals(o.factory);
        }
    }
    
    public int hashCode() {
        FactoryDescription desc = description;
        return desc != null ? desc.hashCode() : factory.hashCode();
    }
    
    public boolean sharesSource(Tool other) {
        if(!(other instanceof AddTool)) return false;
        AddTool o = (AddTool) other;
        if(this.sourceLoadAttempted && o.sourceLoadAttempted) {
            return this.factory.equals(o.factory);
        } else {
            return this.description != null && o.description != null && this.description.equals(o.description);
        }
    }
    
    public ComponentFactory getFactory(boolean forceLoad) {
        return forceLoad ? getFactory() : factory;
    }

    public ComponentFactory getFactory() {
        ComponentFactory ret = factory;
        if(ret != null || sourceLoadAttempted) {
            return ret;
        } else {
            ret = description.getFactory(descriptionBase);
            if(ret != null) {
                Boolean value = (Boolean) ret.getFeature(ComponentFactory.SHOULD_SNAP, attrs);
                shouldSnap = value == null ? true : value.booleanValue();
            }
            factory = ret;
            sourceLoadAttempted = true;
            return ret;
        }
    }

    public String getName() {
        FactoryDescription desc = description;
        return desc == null ? factory.getName() : desc.getName();
    }

    public String getDisplayName() {
        FactoryDescription desc = description;
        return desc == null ? factory.getDisplayName() : desc.getDisplayName();
    }

    public String getDescription() {
        String ret;
        FactoryDescription desc = description;
        if(desc != null) {
            ret = desc.getToolTip();
        } else {
            ComponentFactory source = getFactory();
            if(source != null) {
                ret = (String) source.getFeature(ComponentFactory.TOOL_TIP,
                        getAttributeSet());
            } else {
                ret = null;
            }
        }
        if(ret == null) {
            ret = StringUtil.format(Strings.get("addToolText"), getDisplayName());
        }
        return ret;
    }

    public Tool cloneTool() {
        return new AddTool(this);
    }

    public AttributeSet getAttributeSet() {
        return attrs;
    }

    public void draw(Canvas canvas, ComponentDrawContext context) {
        // next "if" suggested roughly by Kevin Walsh of Cornell to take care of
        // repaint problems on OpenJDK under Ubuntu
        if(lastX == Integer.MAX_VALUE || lastY == Integer.MAX_VALUE) return;
        ComponentFactory source = getFactory();
        if(source == null) return;
        if(state == SHOW_GHOST) {
            source.drawGhost(context,
                Color.GRAY, lastX, lastY, attrs);
        } else if(state == SHOW_ADD) {
            source.drawGhost(context,
                Color.BLACK, lastX, lastY, attrs);
        }
    }

    public void cancelOp() { }

    public void select(Canvas canvas) {
        setState(canvas, SHOW_GHOST);
        bounds = null;
    }

    public void deselect(Canvas canvas) {
        setState(canvas, SHOW_GHOST);
        moveTo(canvas, canvas.getGraphics(),
            Integer.MAX_VALUE, Integer.MAX_VALUE);
        bounds = null;
        lastAddition = null;
    }

    private synchronized void moveTo(Canvas canvas, Graphics g,
            int x, int y) {
        if(state != SHOW_NONE) expose(canvas, lastX, lastY);
        lastX = x;
        lastY = y;
        if(state != SHOW_NONE) expose(canvas, lastX, lastY);
    }

    public void mouseEntered(Canvas canvas, Graphics g,
            MouseEvent e) {
        if(state == SHOW_GHOST || state == SHOW_NONE) {
            setState(canvas, SHOW_GHOST);
            canvas.requestFocusInWindow();
        } else if(state == SHOW_ADD_NO) {
            setState(canvas, SHOW_ADD);
            canvas.requestFocusInWindow();
        }
    }

    public void mouseExited(Canvas canvas, Graphics g,
            MouseEvent e) {
        if(state == SHOW_GHOST) {
            moveTo(canvas, canvas.getGraphics(),
                Integer.MAX_VALUE, Integer.MAX_VALUE);
            setState(canvas, SHOW_NONE);
        } else if(state == SHOW_ADD) {
            moveTo(canvas, canvas.getGraphics(),
                Integer.MAX_VALUE, Integer.MAX_VALUE);
            setState(canvas, SHOW_ADD_NO);
        }
    }

    public void mouseMoved(Canvas canvas, Graphics g, MouseEvent e) {
        if(state != SHOW_NONE) {
            if(shouldSnap) Canvas.snapToGrid(e);
            moveTo(canvas, g, e.getX(), e.getY());
        }
    }

    public void mousePressed(Canvas canvas, Graphics g, MouseEvent e) {
        // verify the addition would be valid
        Circuit circ = canvas.getCircuit();
        if(!canvas.getProject().getLogisimFile().contains(circ)) {
            canvas.setErrorMessage(Strings.getter("cannotModifyError"));
            return;
        }
        if(factory instanceof Circuit) {
            Dependencies depends = canvas.getProject().getDependencies();
            if(!depends.canAdd(circ, (Circuit) factory)) {
                canvas.setErrorMessage(Strings.getter("circularError"));
                return;
            }
        }

        if(shouldSnap) Canvas.snapToGrid(e);
        moveTo(canvas, g, e.getX(), e.getY());
        setState(canvas, SHOW_ADD);
    }

    public void mouseDragged(Canvas canvas, Graphics g, MouseEvent e) {
        if(state != SHOW_NONE) {
            if(shouldSnap) Canvas.snapToGrid(e);
            moveTo(canvas, g, e.getX(), e.getY());
        }
    }

    public void mouseReleased(Canvas canvas, Graphics g,
            MouseEvent e) {
        Component added = null;
        if(state == SHOW_ADD) {
            Circuit circ = canvas.getCircuit();
            if(!canvas.getProject().getLogisimFile().contains(circ)) return;
            if(shouldSnap) Canvas.snapToGrid(e);
            moveTo(canvas, g, e.getX(), e.getY());

            Location loc = Location.create(e.getX(), e.getY());
            AttributeSet attrsCopy = (AttributeSet) attrs.clone();
            ComponentFactory source = getFactory();
            if(source == null) return;
            Component c = source.createComponent(loc, attrsCopy);
            
            if(circ.hasConflict(c)) {
                canvas.setErrorMessage(Strings.getter("exclusiveError"));
                return;
            }
            
            Bounds bds = c.getBounds(g);
            if(bds.getX() < 0 || bds.getY() < 0) {
                canvas.setErrorMessage(Strings.getter("negativeCoordError"));
                return;
            }

            try {
                lastAddition = CircuitActions.addComponent(circ, c, false);
                canvas.getProject().doAction(lastAddition);
                added = c;
            } catch(CircuitException ex) {
                JOptionPane.showMessageDialog(canvas.getProject().getFrame(),
                    ex.getMessage());
            }
            setState(canvas, SHOW_GHOST);
        } else if(state == SHOW_ADD_NO) {
            setState(canvas, SHOW_NONE);
        }
        
        Project proj = canvas.getProject();
        Tool next = determineNext(proj);
        if(next != null) {
            proj.setTool(next);
            canvas.getSelection().clear();
            if(added != null) canvas.getSelection().add(added);
        }
    }
    
    private Tool determineNext(Project proj) {
        String afterAdd = LogisimPreferences.getAfterAdd();
        if(afterAdd.equals(LogisimPreferences.AFTER_ADD_UNCHANGED)) {
            return null;
        } else { // switch to Edit Tool
            Library base = proj.getLogisimFile().getLibrary("Base");
            if(base == null) {
                return null;
            } else {
                return base.getTool("Edit Tool");
            }
        }
    }
    
    public void keyPressed(Canvas canvas, KeyEvent event) {
        switch(event.getKeyCode()) {
        case KeyEvent.VK_UP:    setFacing(canvas, Direction.NORTH); break;
        case KeyEvent.VK_DOWN:  setFacing(canvas, Direction.SOUTH); break;
        case KeyEvent.VK_LEFT:  setFacing(canvas, Direction.WEST); break;
        case KeyEvent.VK_RIGHT: setFacing(canvas, Direction.EAST); break;
        case KeyEvent.VK_BACK_SPACE:
            if(lastAddition != null && canvas.getProject().getLastAction() == lastAddition) {
                canvas.getProject().undoAction();
                lastAddition = null;
            }
        }
    }
    
    private void setFacing(Canvas canvas, Direction facing) {
        ComponentFactory source = getFactory();
        if(source == null) return;
        Attribute attr = (Attribute) source.getFeature(ComponentFactory.FACING_ATTRIBUTE_KEY, attrs);
        if(attr != null) {
            attrs.setValue(attr, facing);
            canvas.repaint();
        }
    }

    public void paintIcon(ComponentDrawContext c, int x, int y) {
        FactoryDescription desc = description;
        if(desc != null) {
            Icon icon = desc.getIcon();
            if(icon != null) {
                icon.paintIcon(c.getDestination(), c.getGraphics(), x + 2, y + 2);
                return;
            }
        }
        
        ComponentFactory source = getFactory();
        if(source != null) source.paintIcon(c, x, y, attrs);
    }

    private void expose(java.awt.Component c, int x, int y) {
        Bounds bds = getBounds();
        c.repaint(x + bds.getX(), y + bds.getY(),
            bds.getWidth(), bds.getHeight());
    }

    public Cursor getCursor() { return cursor; }

    private void setState(Canvas canvas, int value) {
        if(value == SHOW_GHOST) {
            if(canvas.getProject().getLogisimFile().contains(canvas.getCircuit())
                    && LogisimPreferences.getShowGhosts()) {
                state = SHOW_GHOST;
            } else {
                state = SHOW_NONE;
            }
        } else{
            state = value;
        }
    }

    private Bounds getBounds() {
        Bounds ret = bounds;
        if(ret == null) {
            ComponentFactory source = getFactory();
            if(source == null) ret = Bounds.EMPTY_BOUNDS;
            else ret = source.getOffsetBounds(attrs).expand(5);
            bounds = ret;
        }
        return ret;
    }
}
