/* Copyright (c) 2006, 2009, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */
 
package com.cburch.logisim.std.io;

import java.awt.Color;
import java.awt.Graphics;

import com.cburch.logisim.data.Attribute;
import com.cburch.logisim.data.AttributeSet;
import com.cburch.logisim.data.Bounds;
import com.cburch.logisim.data.Direction;
import com.cburch.logisim.data.Value;
import com.cburch.logisim.instance.Instance;
import com.cburch.logisim.instance.InstanceDataSingleton;
import com.cburch.logisim.instance.InstanceFactory;
import com.cburch.logisim.instance.InstanceLogger;
import com.cburch.logisim.instance.InstancePainter;
import com.cburch.logisim.instance.InstanceState;
import com.cburch.logisim.instance.Port;
import com.cburch.logisim.instance.StdAttr;
import com.cburch.logisim.util.GraphicsUtil;

public class Led extends InstanceFactory {
    public Led() {
        super("LED", Strings.getter("ledComponent"));
        setAttributes(new Attribute[] {
                StdAttr.FACING, Io.ATTR_COLOR,
                StdAttr.LABEL, Io.ATTR_LABEL_LOC,
                StdAttr.LABEL_FONT, Io.ATTR_LABEL_COLOR
            }, new Object[] {
                Direction.WEST, new Color(240, 0, 0),
                "", Io.LABEL_CENTER,
                StdAttr.DEFAULT_LABEL_FONT, Color.BLACK
            });
        setFacingAttribute(StdAttr.FACING);
        setIconName("led.gif");
        setPorts(new Port[] { new Port(0, 0, Port.INPUT, 1) });
        setInstanceLogger(Logger.class);
    }

    public Bounds getOffsetBounds(AttributeSet attrs) {
        Direction facing = (Direction) attrs.getValue(StdAttr.FACING);
        return Bounds.create(0, -10, 20, 20).rotate(Direction.WEST, facing, 0, 0);
    }

    protected void configureNewInstance(Instance instance) {
        instance.addAttributeListener();
        computeTextField(instance);
    }

    protected void instanceAttributeChanged(Instance instance, Attribute attr) {
        if(attr == StdAttr.FACING) {
            instance.recomputeBounds();
            computeTextField(instance);
        } else if(attr == Io.ATTR_LABEL_LOC) {
            computeTextField(instance);
        }
    }

    private void computeTextField(Instance instance) {
        Direction facing = (Direction) instance.getAttributeValue(StdAttr.FACING);
        Object labelLoc = instance.getAttributeValue(Io.ATTR_LABEL_LOC);

        Bounds bds = instance.getBounds();
        int x = bds.getX() + bds.getWidth() / 2;
        int y = bds.getY() + bds.getHeight() / 2;
        int halign = GraphicsUtil.H_CENTER;
        int valign = GraphicsUtil.V_CENTER;
        if(labelLoc == Direction.NORTH) {
            y = bds.getY() - 2;
            valign = GraphicsUtil.V_BOTTOM;
        } else if(labelLoc == Direction.SOUTH) {
            y = bds.getY() + bds.getHeight() + 2;
            valign = GraphicsUtil.V_TOP;
        } else if(labelLoc == Direction.EAST) {
            x = bds.getX() + bds.getWidth() + 2;
            halign = GraphicsUtil.H_LEFT;
        } else if(labelLoc == Direction.WEST) {
            x = bds.getX() - 2;
            halign = GraphicsUtil.H_RIGHT;
        }
        if(labelLoc == facing) {
            if(labelLoc == Direction.NORTH || labelLoc == Direction.SOUTH) {
                x += 2;
                halign = GraphicsUtil.H_LEFT;
            } else {
                y -= 2;
                valign = GraphicsUtil.V_BOTTOM;
            }
        }

        instance.setTextField(StdAttr.LABEL, StdAttr.LABEL_FONT,
                x, y, halign, valign);
    }

    public void propagate(InstanceState state) {
        Value val = state.getPort(0);
        InstanceDataSingleton data = (InstanceDataSingleton) state.getData();
        if(data == null) {
            state.setData(new InstanceDataSingleton(val));
        } else {
            data.setValue(val);
        }
    }
    
    public void paintGhost(InstancePainter painter) {
        Graphics g = painter.getGraphics();
        Bounds bds = painter.getBounds();
        GraphicsUtil.switchToWidth(g, 2);
        g.drawOval(bds.getX() + 1, bds.getY() + 1,
                bds.getWidth() - 2, bds.getHeight() - 2);
    }

    public void paintInstance(InstancePainter painter) {
        InstanceDataSingleton data = (InstanceDataSingleton) painter.getData();
        Value val = data == null ? Value.FALSE : (Value) data.getValue();
        Color color = (Color) painter.getAttributeValue(Io.ATTR_COLOR);
        Bounds bds = painter.getBounds().expand(-1);

        Graphics g = painter.getGraphics();
        if(painter.getShowState()) {
            g.setColor(val == Value.TRUE ? color : Color.DARK_GRAY);
            g.fillOval(bds.getX(), bds.getY(), bds.getWidth(), bds.getHeight());
        }
        g.setColor(Color.BLACK);
        GraphicsUtil.switchToWidth(g, 2);
        g.drawOval(bds.getX(), bds.getY(), bds.getWidth(), bds.getHeight());
        GraphicsUtil.switchToWidth(g, 1);
        g.setColor((Color) painter.getAttributeValue(Io.ATTR_LABEL_COLOR));
        painter.drawLabel();
        painter.drawPorts();
    }

    public static class Logger extends InstanceLogger {
        public String getLogName(InstanceState state, Object option) {
            return (String) state.getAttributeValue(StdAttr.LABEL);
        }
    
        public Value getLogValue(InstanceState state, Object option) {
            InstanceDataSingleton data = (InstanceDataSingleton) state.getData();
            if(data == null) return Value.FALSE;
            return data.getValue() == Value.TRUE ? Value.TRUE : Value.FALSE; 
	 }
    }
}
