/* Copyright (c) 2006, 2009, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */
 
package com.cburch.logisim.std.gates;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.util.Map;

import javax.swing.Icon;

import com.cburch.logisim.analyze.model.Expression;
import com.cburch.logisim.circuit.CircuitState;
import com.cburch.logisim.circuit.ExpressionComputer;
import com.cburch.logisim.comp.AbstractComponentFactory;
import com.cburch.logisim.comp.Component;
import com.cburch.logisim.comp.ComponentFactory;
import com.cburch.logisim.comp.ComponentDrawContext;
import com.cburch.logisim.comp.EndData;
import com.cburch.logisim.comp.ManagedComponent;
import com.cburch.logisim.data.Attribute;
import com.cburch.logisim.data.AttributeEvent;
import com.cburch.logisim.data.AttributeListener;
import com.cburch.logisim.data.AttributeSet;
import com.cburch.logisim.data.AttributeSets;
import com.cburch.logisim.data.BitWidth;
import com.cburch.logisim.data.Bounds;
import com.cburch.logisim.data.Direction;
import com.cburch.logisim.data.Location;
import com.cburch.logisim.data.Value;
import com.cburch.logisim.file.Options;
import com.cburch.logisim.instance.StdAttr;
import com.cburch.logisim.util.GraphicsUtil;
import com.cburch.logisim.util.Icons;

class Buffer extends ManagedComponent
        implements AttributeListener, ExpressionComputer {
    public static ComponentFactory FACTORY = new Factory();

    private static final Attribute[] ATTRIBUTES
        = { StdAttr.FACING, StdAttr.WIDTH };
    
    private static final Object[] DEFAULTS = { Direction.EAST, BitWidth.ONE };
    private static final Icon toolIcon = Icons.getIcon("bufferGate.gif");

    private static class Factory extends AbstractComponentFactory {
        private Factory() { }

        public String getName() { return "Buffer"; }

        public String getDisplayName() { return Strings.get("bufferComponent"); }

        public AttributeSet createAttributeSet() {
            return AttributeSets.fixedSet(ATTRIBUTES, DEFAULTS);
        }

        public Component createComponent(Location loc, AttributeSet attrs) {
            return new Buffer(loc, attrs);
        }

        public Bounds getOffsetBounds(AttributeSet attrs) {
            Direction facing = (Direction) attrs.getValue(StdAttr.FACING);
            if(facing == Direction.SOUTH) return Bounds.create(-9, -20, 18, 20);
            if(facing == Direction.NORTH) return Bounds.create(-9, 0, 18, 20);
            if(facing == Direction.WEST) return Bounds.create(0, -9, 20, 18);
            return Bounds.create(-20, -9, 20, 18);
        }
        
        //
        // user interface methods
        //
        public void drawGhost(ComponentDrawContext context,
                Color color, int x, int y, AttributeSet attrs) {
            Graphics g = context.getGraphics();
            g.setColor(color);
            drawBase(g, attrs, x, y);
        }

        public void paintIcon(ComponentDrawContext context,
                int x, int y, AttributeSet attrs) {
            Graphics g = context.getGraphics();
            g.setColor(Color.black);
            if(toolIcon != null) {
                toolIcon.paintIcon(context.getDestination(), g, x + 2, y + 2);
            } else {
                g.setColor(Color.black);
                int[] xp = new int[4];
                int[] yp = new int[4];
                xp[0] = x + 17; yp[0] = y + 10;
                xp[1] = x +  3; yp[1] = y +  3;
                xp[2] = x +  3; yp[2] = y + 17;
                xp[3] = x + 17; yp[3] = y + 10;
                g.drawPolyline(xp, yp, 4);
            }
        }

        public Object getFeature(Object key, AttributeSet attrs) {
            if(key == FACING_ATTRIBUTE_KEY) return StdAttr.FACING;
            return super.getFeature(key, attrs);
        }
    }

    public Buffer(Location loc, AttributeSet attrs) {
        super(loc, attrs, 2);
        attrs.addAttributeListener(this);
        setPins();
    }

    private void setPins() {
        AttributeSet attrs = getAttributeSet();
        Direction dir = (Direction) attrs.getValue(StdAttr.FACING);
        BitWidth w = (BitWidth) attrs.getValue(StdAttr.WIDTH);
        Location loc0 = getLocation();
        Location loc1 = loc0.translate(dir.reverse(), 20);
        
        EndData[] ends = new EndData[2];
        ends[0] = new EndData(loc0, w, EndData.OUTPUT_ONLY);
        ends[1] = new EndData(loc1, w, EndData.INPUT_ONLY);
        setEnds(ends);
    }

    public ComponentFactory getFactory() {
        return FACTORY;
    }

    public void propagate(CircuitState state) {
        Value in = state.getValue(getEndLocation(1));
        in = Buffer.repair(state, this, in);
        state.setValue(getEndLocation(0), in, this, GateAttributes.DELAY);
    }

    public void attributeListChanged(AttributeEvent e) { }
    public void attributeValueChanged(AttributeEvent e) {
        Attribute attr = e.getAttribute();
        if(attr == StdAttr.FACING) {
            recomputeBounds();
            setPins();
        } else if(attr == StdAttr.WIDTH) {
            setPins();
        }
    }
    
    //
    // user interface methods
    //
    public void draw(ComponentDrawContext context) {
        Graphics g = context.getGraphics();
        Location loc = getLocation();
        int x = loc.getX();
        int y = loc.getY();

        // draw gate
        g.setColor(Color.BLACK);
        drawBase(g, getAttributeSet(), x, y);
        context.drawPins(this);
    }
    
    public Object getFeature(Object key) {
        if(key == ExpressionComputer.class) return this;
        return super.getFeature(key);
    }

    public void computeExpression(Map expressionMap) {
        Expression e = (Expression) expressionMap.get(getEndLocation(1));
        if(e != null) {
            expressionMap.put(getEndLocation(0), e);
        }
    }
    
    static Value repair(CircuitState state, Component gate, Value v) {
        AttributeSet opts = state.getProject().getOptions().getAttributeSet();
        boolean errorIfUndefined = opts.getValue(Options.ATTR_GATE_UNDEFINED)
            .equals(Options.GATE_UNDEFINED_ERROR);
        if(errorIfUndefined) {
            int vw = v.getWidth();
            BitWidth w = (BitWidth) gate.getAttributeSet().getValue(StdAttr.WIDTH);
            int ww = w.getWidth();
            if(vw == ww && v.isFullyDefined()) return v;
            Value[] vs = new Value[w.getWidth()];
            for(int i = 0; i < vs.length; i++) {
                Value ini = i < vw ? v.get(i) : Value.ERROR;
                vs[i] = ini.isFullyDefined() ? ini : Value.ERROR;
            }
            return Value.create(vs);
        } else {
            return v;
        }
    }

    private static void drawBase(Graphics oldG, AttributeSet attrs,
            int x, int y) {
        Direction facing = (Direction) attrs.getValue(StdAttr.FACING);
        Graphics g = oldG;
        if(facing != Direction.EAST && oldG instanceof Graphics2D) {
            Graphics2D g2 = (Graphics2D) g.create();
            g2.rotate(-facing.toRadians(), x, y);
            g = g2;
        }

        GraphicsUtil.switchToWidth(g, 2);
        int[] xp = new int[4];
        int[] yp = new int[4];
        xp[0] = x;      yp[0] = y;
        xp[1] = x - 19; yp[1] = y - 7;
        xp[2] = x - 19; yp[2] = y + 7;
        xp[3] = x;      yp[3] = y;
        g.drawPolyline(xp, yp, 4);
    }
}
