/* Copyright (c) 2006, 2009, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */
 
package com.cburch.logisim.std.gates;

import java.awt.Color;
import java.awt.Graphics;

import javax.swing.Icon;

import com.cburch.logisim.analyze.model.Expression;
import com.cburch.logisim.comp.AbstractComponentFactory;
import com.cburch.logisim.comp.Component;
import com.cburch.logisim.comp.ComponentDrawContext;
import com.cburch.logisim.data.AttributeOption;
import com.cburch.logisim.data.AttributeSet;
import com.cburch.logisim.data.Bounds;
import com.cburch.logisim.data.Direction;
import com.cburch.logisim.data.Location;
import com.cburch.logisim.data.Value;
import com.cburch.logisim.instance.StdAttr;
import com.cburch.logisim.proj.LogisimPreferences;
import com.cburch.logisim.tools.WireRepairData;
import com.cburch.logisim.util.GraphicsUtil;
import com.cburch.logisim.util.StringGetter;

abstract class AbstractGateFactory extends AbstractComponentFactory {
    private static final int ATTEMPT_SHAPED = 1;
    private static final int ATTEMPT_RECTANGULAR = 2;
    private static final int ATTEMPT_DIN40700 = 4;
    
    private String name;
    private StringGetter desc;
    private int iconAttempts = 0;
    private Icon iconShaped = null;
    private Icon iconRect = null;
    private Icon iconDin = null;
    int bonusWidth = 0;
    boolean hasDongle = false;
    String rectLabel = "";
    private boolean isXor;

    protected AbstractGateFactory(String name, StringGetter desc) {
        this(name, desc, false);
    }
    
    protected AbstractGateFactory(String name, StringGetter desc, boolean isXor) {
        this.name = name;
        this.desc = desc;
        this.isXor = isXor;
    }

    public String getName() { return name; }

    public String getDisplayName() { return desc.get(); }

    public AttributeSet createAttributeSet() {
        return new GateAttributes(isXor);
    }

    public Component createComponent(Location loc, AttributeSet attrs) {
        return new AbstractGate(loc, attrs, this);
    }

    public Bounds getOffsetBounds(AttributeSet attrs) {
        return computeBounds(attrs);
    }

    //
    // user interface methods
    //
    public void drawGhost(ComponentDrawContext context,
            Color color, int x, int y, AttributeSet attrs) {
        Bounds bounds = computeBounds(attrs);
        context.getGraphics().setColor(color);
        AbstractGate.drawBase(context, this, null, attrs, x, y,
                bounds.getWidth(), bounds.getHeight());
    }
    
    public Object getFeature(Object key, AttributeSet attrs) {
        if(key == FACING_ATTRIBUTE_KEY) return StdAttr.FACING;
        return super.getFeature(key, attrs);
    }
    
    public abstract Icon getIconShaped();
    public abstract Icon getIconRectangular();
    public abstract Icon getIconDin40700();
    public abstract void paintIconShaped(ComponentDrawContext context,
            int x, int y, AttributeSet attrs);
    public void paintIconRectangular(ComponentDrawContext context,
            int x, int y, AttributeSet attrs) {
        Graphics g = context.getGraphics();
        g.drawRect(x + 1, y + 2, 16, 16);
        if(hasDongle) g.drawOval(x + 16, y + 8, 4, 4);
        GraphicsUtil.drawCenteredText(g, getRectangularLabel(attrs), x + 9, y + 8);
    }

    public final void paintIcon(ComponentDrawContext context,
            int x, int y, AttributeSet attrs) {
        Graphics g = context.getGraphics();
        g.setColor(Color.black);
        if(context.getGateShape() == LogisimPreferences.SHAPE_RECTANGULAR) {
            if(iconRect == null && (iconAttempts & ATTEMPT_RECTANGULAR) == 0) {
                iconRect = getIconRectangular();
                iconAttempts |= ATTEMPT_RECTANGULAR;
            }
            if(iconRect != null) {
                iconRect.paintIcon(context.getDestination(), g, x + 2, y + 2);
            } else {
                paintIconRectangular(context, x, y, attrs);
            }
        } else if(context.getGateShape() == LogisimPreferences.SHAPE_DIN40700) {
            if(iconDin == null && (iconAttempts & ATTEMPT_DIN40700) == 0) {
                iconDin = getIconDin40700();
                iconAttempts |= ATTEMPT_DIN40700;
            }
            if(iconDin != null) {
                iconDin.paintIcon(context.getDestination(), g, x + 2, y + 2);
            } else {
                paintIconRectangular(context, x, y, attrs);
            }
        } else {
            if(iconShaped == null && (iconAttempts & ATTEMPT_SHAPED) == 0) {
                iconShaped = getIconShaped();
                iconAttempts |= ATTEMPT_SHAPED;
            }
            if(iconShaped != null) {
                iconShaped.paintIcon(context.getDestination(), g, x + 2, y + 2);
            } else {
                paintIconShaped(context, x, y, attrs);
            }
        }
    }

    protected void setAdditionalWidth(int value) {
        bonusWidth = value;
    }

    protected void setHasDongle(boolean value) {
        hasDongle = value;
    }

    protected void setRectangularLabel(String value) {
        rectLabel = value;
    }

    protected String getRectangularLabel(AttributeSet attrs) {
        return rectLabel;
    }

    //
    // protected methods intended to be overridden
    //
    protected void drawInputLines(ComponentDrawContext context,
            AbstractGate comp, int inputs,
            int x, int yTop, int width, int height) {
    }

    protected abstract void drawShape(ComponentDrawContext context,
            AbstractGate gate, int x, int y, int width, int height);

    protected void drawRectangular(ComponentDrawContext context,
            AbstractGate gate, int x, int y, int width, int height) {
        int don = hasDongle ? 10 : 0;
        AttributeSet attrs = gate == null ? null : gate.getAttributeSet();
        context.drawRectangle(x - width, y - height / 2, width - don, height,
                getRectangularLabel(attrs));
        if(hasDongle) {
            context.drawDongle(x - 5, y);
        }
    }

    protected abstract void drawDinShape(ComponentDrawContext context,
            AbstractGate gate, int x, int y, int width, int height, int inputs);
    
    protected abstract Value computeOutput(Value[] inputs,
            int num_inputs, AttributeSet attrs);
    
    protected abstract Expression computeExpression(Expression[] inputs,
            int numInputs);

    protected boolean shouldRepairWire(Component comp, WireRepairData data) {
        return false;
    }

    //
    // private helper methods
    //
    private Bounds computeBounds(AttributeSet attrs) {
        Direction facing = (Direction) attrs.getValue(StdAttr.FACING);
        AttributeOption sizeOpt = (AttributeOption) attrs.getValue(GateAttributes.ATTR_SIZE);
        int size = ((Integer) sizeOpt.getValue()).intValue();
        int inputs = ((Integer) attrs.getValue(GateAttributes.ATTR_INPUTS)).intValue();
        
        int width = size + bonusWidth + (hasDongle ? 10 : 0);
        int height = Math.max(10 * inputs, size);
        if(facing == Direction.SOUTH) {
            return Bounds.create(-height / 2, -width, height, width);
        } else if(facing == Direction.NORTH) {
            return Bounds.create(-height / 2, 0, height, width);
        } else if(facing == Direction.WEST) {
            return Bounds.create(0, -height / 2, width, height);
        } else {
            return Bounds.create(-width, -height / 2, width, height);
        }
    }
}
