/* Copyright (c) 2006, 2009, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */
 
package com.cburch.logisim.gui.log;

import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.File;
import java.util.Date;

import com.cburch.logisim.data.Value;
import com.cburch.logisim.data.BitWidth;

class LogThread implements ModelListener {
    // file will be flushed with at least this frequency
    private static final int FLUSH_FREQUENCY = 500;
    
    // file will be closed after waiting this many milliseconds between writes
    private static final int IDLE_UNTIL_CLOSE = 10000;

    private class SyncTask implements Runnable {
	boolean canceled;
	public void run() {
	    while(true) {
		synchronized(lock) {
		    if (canceled) return;
		    if(writer != null && System.currentTimeMillis() - lastWrite > IDLE_UNTIL_CLOSE)
			close();
		    else if(writer != null)
			writer.flush();
		}
		try { Thread.sleep(FLUSH_FREQUENCY); } catch(InterruptedException e) { }
	    }
	}
    }
    
    private Model model;
    private LogFrame logFrame;
    private Object lock = new Object();
    private PrintWriter writer = null;
    private boolean headerDirty = true;
    private int entriesDirty = 0;
    private File lastFile;
    private String lastHeader;
    private long lastWrite = 0;
    private SyncTask syncTask;
    
    public LogThread(LogFrame frame, Model model) {
        this.model = model;
	this.logFrame = frame;
        model.addModelListener(this);
    }

    public void start() {
	syncTask = new SyncTask();
	new Thread(syncTask).start();
    }
    
    public void cancel() {
	if (syncTask != null) {
	    synchronized(lock) {
		syncTask.canceled = true;
		syncTask = null;
		close();
	    }
	}
    }

    private void close() {
        synchronized(lock) {
            if(writer != null) {
                writer.close();
                writer = null;
            }
        }
    }

    private boolean ready() {
        return model.isFileEnabled() && model.getFile() != null;
    }
    

    public void selectionChanged(ModelEvent event) {
        headerDirty = true;
    }

    public void entryAdded(ModelEvent event, Value[] values) {
        synchronized(lock) {
            if(ready() && model.isFileKeepWriting()) addEntry(values);
	    else entriesDirty++;
        }
    }

    public void filePropertyChanged(ModelEvent event) {
        synchronized(lock) {
            if(!ready() || !model.isSelected()) {
		close();
		return;
	    }
	    if(entriesDirty > 0) {
		Selection sel = model.getSelection();
		ValueLog[] logs = new ValueLog[sel.size()];
		int logsize = 0;
		Value[] values = new Value[sel.size()];
		for(int i = 0; i < values.length; i++) {
		    logs[i] = model.getValueLog(sel.get(i));
		    logsize = Math.max(logsize, logs[i].size());
		}
		if (logsize > entriesDirty)
		    logsize = entriesDirty;
		entriesDirty = 0;
		for (int j = logsize; j >= 1; j--) {
		    for(int i = 0; i < values.length; i++) {
			if (logs[i].size() >= j) values[i] = logs[i].get(logs[i].size() - j);
			else values[i] = null;
		    }
		    addEntry(values);
		}
	    }
	    if (!model.isFileKeepWriting())
		close();
        }
    }
    
    // Should hold lock and have verified that ready() before entering this method.
    private void addEntry(Value[] values) {
        if(writer == null) {
            try {
		File file = model.getFile();
		if (!file.equals(lastFile)) {
		    headerDirty = true;
		    lastHeader = null;
		}
                writer = new PrintWriter(new FileWriter(file, true));
            } catch(IOException e) {
                model.setFile(e);
                return;
            }
        }
        Selection sel = model.getSelection();
        if(headerDirty) {
            if(model.getFileHeader()) {
                StringBuffer buf = new StringBuffer();
                for(int i = 0; i < sel.size(); i++) {
                    if(i > 0) buf.append("\t");
                    buf.append(sel.get(i).toString());
		    BitWidth width = sel.get(i).getWidth();
		    if (width.getWidth() > 1)
			buf.append("[" + width.getWidth() + "]");
                }
		String header = buf.toString();
		if (!header.equals(lastHeader)) {
		    lastFile = model.getFile();
		    lastHeader = header;
		    writer.println("# " + LogFrame.computeTitle(model, logFrame.getProject()));
		    writer.println("# " + new Date());
		    writer.println(header);
		}
            }
            headerDirty = false;
        }
        StringBuffer buf = new StringBuffer();
        for(int i = 0; i < values.length; i++) {
            if(i > 0) buf.append("\t");
            if(values[i] != null) {
                int radix = sel.get(i).getRadix();
                buf.append(values[i].toLogString(radix));
            }
        }
        writer.println(buf.toString());
        lastWrite = System.currentTimeMillis();
    }
}
