/* Copyright (c) 2006, 2009, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */
 
package com.cburch.draw.undo;

import java.util.Iterator;
import java.util.LinkedList;

import com.cburch.logisim.util.EventSourceWeakSupport;

public class UndoLog {
    private static final int MAX_UNDO_SIZE = 64;

    private EventSourceWeakSupport listeners;
    private LinkedList undoLog;
    private LinkedList redoLog;
    private int modCount;

    public UndoLog() {
        this.listeners = new EventSourceWeakSupport();
        this.undoLog = new LinkedList();
        this.redoLog = new LinkedList();
        this.modCount = 0;
    }
    
    //
    // listening methods
    //
    public void addProjectListener(UndoLogListener what) {
        listeners.add(what);
    }

    public void removeProjectListener(UndoLogListener what) {
        listeners.remove(what);
    }

    private void fireEvent(int action, Action actionObject) {
        UndoLogEvent e = null;
        for(Iterator it = listeners.iterator(); it.hasNext(); ) {
            UndoLogListener l = (UndoLogListener) it.next();
            if(e == null) e = new UndoLogEvent(this, action, actionObject);
            l.undoLogChanged(e);
        }
    }

    //
    // accessor methods
    //
    public Action getUndoAction() {
        if(undoLog.size() == 0) {
            return null;
        } else {
            return (Action) undoLog.getLast();
        }
    }

    public Action getRedoAction() {
        if(redoLog.size() == 0) {
            return null;
        } else {
            return (Action) redoLog.getLast();
        }
    }

    public boolean isModified() {
        return modCount != 0;
    }

    //
    // mutator methods
    //
    public void doAction(Action act) {
        if(act == null) return;
        act.doIt();
        logAction(act);
    }
    
    public void logAction(Action act) {
        redoLog.clear();
        if(!undoLog.isEmpty()) {
            Action prev = (Action) undoLog.getLast();
            if(act.shouldAppendTo(prev)) {
                if(prev.isModification()) --modCount;
                Action joined = prev.append(act);
                if(joined == null) {
                    fireEvent(UndoLogEvent.ACTION_DONE, act);
                    return;
                }
                act = joined;
            }
            while(undoLog.size() > MAX_UNDO_SIZE) {
                undoLog.removeFirst();
            }
        }
        undoLog.add(act);
        if(act.isModification()) ++modCount;
        fireEvent(UndoLogEvent.ACTION_DONE, act);
    }

    public void undoAction() {
        if(undoLog.size() > 0) {
            Action action = (Action) undoLog.removeLast();
            if(action.isModification()) --modCount;
            action.undo();
            redoLog.add(action);
            fireEvent(UndoLogEvent.ACTION_UNDONE, action);
        }
    }

    public void redoAction() {
        if(redoLog.size() > 0) {
            Action action = (Action) redoLog.removeLast();
            if(action.isModification()) ++modCount;
            action.doIt();
            undoLog.add(action);
            fireEvent(UndoLogEvent.ACTION_DONE, action);
        }
    }

    public void clearModified() {
        modCount = 0;
    }
}
