/* Copyright (c) 2006, 2009, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */
 
package com.cburch.draw.tools;

import java.awt.Color;
import java.awt.Cursor;
import java.awt.Graphics;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.util.List;

import javax.swing.Icon;

import com.cburch.draw.actions.ModelAddAction;
import com.cburch.draw.canvas.Canvas;
import com.cburch.draw.canvas.CanvasModel;
import com.cburch.draw.canvas.CanvasObject;
import com.cburch.draw.model.Drawables;
import com.cburch.draw.model.DrawingAttribute;
import com.cburch.draw.model.DrawingAttributeSet;
import com.cburch.logisim.data.Location;
import com.cburch.logisim.util.Icons;

class LineTool extends AbstractTool {
    private DrawingAttributeSet attrs;
    private boolean active;
    private Location mouseStart;
    private Location mouseEnd;
    private int lastMouseX;
    private int lastMouseY;
    
    public LineTool(DrawingAttributeSet attrs) {
        this.attrs = attrs;
        active = false;
    }
    
    public Icon getIcon() {
        return Icons.getIcon("drawline.gif");
    }

    public Cursor getCursor(Canvas canvas) {
        return Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR);
    }
    
    public List getAttributes() {
        return DrawingAttribute.ATTRS_STROKE;
    }
    
    public void toolDeselected(Canvas canvas) {
        active = false;
        repaintArea(canvas);
    }
    
    public void mousePressed(Canvas canvas, MouseEvent e) {
        Location loc = Location.create(e.getX(), e.getY());
        mouseStart = loc;
        mouseEnd = loc;
        lastMouseX = loc.getX();
        lastMouseY = loc.getY();
        active = canvas.getModel() != null;
        repaintArea(canvas);
    }
    
    public void mouseDragged(Canvas canvas, MouseEvent e) {
        updateMouse(canvas, e.getX(), e.getY(), e.getModifiersEx());
    }
    
    public void mouseReleased(Canvas canvas, MouseEvent e) {
        if(active) {
            updateMouse(canvas, e.getX(), e.getY(), e.getModifiersEx());
            Location start = mouseStart;
            Location end = mouseEnd;
            if(!start.equals(end)) {
                active = false;
                CanvasModel model = canvas.getModel();
                CanvasObject add = Drawables.createLine(start.getX(), start.getY(),
                        end.getX(), end.getY(), attrs);
                model.doAction(new ModelAddAction(model, add));
                repaintArea(canvas);
            }
        }
    }
    
    public void keyPressed(Canvas canvas, KeyEvent e) {
        if(active && e.getKeyCode() == KeyEvent.VK_SHIFT) {
            updateMouse(canvas, lastMouseX, lastMouseY, e.getModifiersEx());
        }
    }
    
    public void keyReleased(Canvas canvas, KeyEvent e) {
        if(active && e.getKeyCode() == KeyEvent.VK_SHIFT) {
            updateMouse(canvas, lastMouseX, lastMouseY, e.getModifiersEx());
        }
    }
    
    private void updateMouse(Canvas canvas, int mx, int my, int mods) {
        if(active) {
            boolean shift = (mods & MouseEvent.SHIFT_DOWN_MASK) != 0;
            Location newEnd = shift ? snapTo8Cardinals(mouseStart, mx, my)
                    : Location.create(mx, my);
            if(!newEnd.equals(mouseEnd)) {
                mouseEnd = newEnd;
                repaintArea(canvas);
            }
        }
        lastMouseX = mx;
        lastMouseY = my;
    }

    private void repaintArea(Canvas canvas) {
        canvas.repaint();
    }
    
    public void draw(Canvas canvas, Graphics g) {
        if(active) {
            Location start = mouseStart;
            Location end = mouseEnd;
            g.setColor(Color.GRAY);
            g.drawLine(start.getX(), start.getY(), end.getX(), end.getY());
        }
    }

    static Location snapTo8Cardinals(Location from, int mx, int my) {
        int px = from.getX();
        int py = from.getY();
        if(mx != px && my != py) {
            double ang = Math.atan2(my - py, mx - px);
            int d45 = (Math.abs(mx - px) + Math.abs(my - py)) / 2;
            int d = (int) (4 * ang / Math.PI + 4.5);
            switch(d) {
            case 0: case 8: // going west
            case 4: // going east
                return Location.create(mx, py);
            case 2: // going north
            case 6: // going south
                return Location.create(px, my);
            case 1: // going northwest
                return Location.create(px - d45, py - d45);
            case 3: // going northeast
                return Location.create(px + d45, py - d45);
            case 5: // going southeast
                return Location.create(px + d45, py + d45);
            case 7: // going southwest
                return Location.create(px - d45, py + d45);
            }
        }
        return Location.create(mx, my); // should never happen
    }
    
    static Location snapTo4Cardinals(Location from, int mx, int my) {
        int px = from.getX();
        int py = from.getY();
        if(mx != px && my != py) {
            if(Math.abs(my - py) < Math.abs(mx - px)) {
                return Location.create(mx, py);
            } else {
                return Location.create(px, my);
            }
        }
        return Location.create(mx, my); // should never happen
    }
}
