/* Copyright (c) 2006, 2009, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */
 
package com.cburch.draw.model;

import com.cburch.draw.canvas.CanvasObject;
import com.cburch.logisim.data.Attribute;
import com.cburch.logisim.data.AttributeEvent;
import com.cburch.logisim.data.AttributeListener;
import com.cburch.logisim.data.AttributeSet;
import com.cburch.logisim.util.EventSourceWeakSupport;

import java.util.Iterator;
import java.util.List;

public abstract class DrawingMember implements AttributeSet, CanvasObject, Cloneable {
    private EventSourceWeakSupport listeners;
    
    public DrawingMember() {
        listeners = new EventSourceWeakSupport();
    }
    
    public AttributeSet getAttributeSet() {
        return this;
    }
    
    protected abstract void setAttrValue(Attribute attr, Object value);
    public abstract void translate(int dx, int dy);
    public abstract void moveHandle(int index, int dx, int dy);
    
    public boolean canInsertHandle(int handleIndex) {
        return false;
    }
    
    public boolean canDeleteHandle(int handleIndex) {
        return false;
    }
    
    public void insertHandle(int handleIndex) {
        throw new UnsupportedOperationException("insertHandle");
    }
    
    public void deleteHandle(int handleIndex) {
        throw new UnsupportedOperationException("deleteHandle");
    }

    // methods required by AttributeSet interface
    public abstract List getAttributes();
    public abstract Object getValue(Attribute attr);

    public void addAttributeListener(AttributeListener l) {
        listeners.add(l);
    }

    public void removeAttributeListener(AttributeListener l) {
        listeners.remove(l);
    }
    
    public Object clone() {
        try {
            DrawingMember ret = (DrawingMember) super.clone();
            ret.listeners = new EventSourceWeakSupport();
            return ret;
        } catch(CloneNotSupportedException e) {
            return null;
        }
    }

    public boolean containsAttribute(Attribute attr) {
        return getAttributes().contains(attr);
    }
    
    public Attribute getAttribute(String name) {
        for(Iterator it = getAttributes().iterator(); it.hasNext(); ) {
            Attribute attr = (Attribute) it.next();
            if(attr.getName().equals(name)) return attr;
        }
        return null;
    }

    public boolean isReadOnly(Attribute attr) {
        return false;
    }
    
    public void setReadOnly(Attribute attr, boolean value) {
        throw new UnsupportedOperationException("setReadOnly");
    }

    public final void setValue(Attribute attr, Object value) {
        Object old = getValue(attr);
        boolean same = old == null ? value == null : old.equals(value);
        if(!same) {
            setAttrValue(attr, value);
            AttributeEvent e = new AttributeEvent(this, attr, value);
            for(Iterator it = listeners.iterator(); it.hasNext(); ) {
                AttributeListener l = (AttributeListener) it.next();
                l.attributeValueChanged(e);
            }
        }
    }
}
