/* Copyright (c) 2006, 2009, Carl Burch. License information is located in the
 * com.cburch.logisim.Main source code and at www.cburch.com/logisim/. */
 
package com.cburch.draw.model;

import java.awt.Graphics;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import com.cburch.draw.canvas.AttributeMapKey;
import com.cburch.draw.canvas.CanvasModel;
import com.cburch.draw.canvas.CanvasModelEvent;
import com.cburch.draw.canvas.CanvasModelListener;
import com.cburch.draw.canvas.CanvasObject;
import com.cburch.draw.canvas.Selection;
import com.cburch.draw.undo.Action;
import com.cburch.draw.undo.UndoLog;
import com.cburch.logisim.data.Attribute;
import com.cburch.logisim.data.Bounds;
import com.cburch.logisim.data.Location;
import com.cburch.logisim.util.EventSourceWeakSupport;

public class Drawing implements CanvasModel {
    private EventSourceWeakSupport listeners;
    private ArrayList canvasObjects;
    private UndoLog undoLog;
    
    public Drawing() {
        listeners = new EventSourceWeakSupport();
        canvasObjects = new ArrayList();
        undoLog = new UndoLog();
    }
    
    public void addCanvasModelListener(CanvasModelListener l) {
        listeners.add(l);
    }
    
    public void removeCanvasModelListener(CanvasModelListener l) {
        listeners.remove(l);
    }

    private void fireChanged(int action, Collection affected,
            int index, int dx, int dy) {
        CanvasModelEvent e = null;
        for(Iterator it = listeners.iterator(); it.hasNext(); ) {
            CanvasModelListener l = (CanvasModelListener) it.next();
            if(e == null) {
                e = new CanvasModelEvent(this, action, affected, index, dx, dy);
            }
            l.modelChanged(e);
        }
    }

    private void fireChanged(int action, CanvasObject affected,
            int index, int dx, int dy) {
        CanvasModelEvent e = null;
        for(Iterator it = listeners.iterator(); it.hasNext(); ) {
            CanvasModelListener l = (CanvasModelListener) it.next();
            if(e == null) {
                Collection coll = Collections.singleton(affected);
                e = new CanvasModelEvent(this, action, coll, index, dx, dy);
            }
            l.modelChanged(e);
        }
    }

    private void fireChanged(int action, Map oldValues, Map newValues) {
        CanvasModelEvent e = null;
        for(Iterator it = listeners.iterator(); it.hasNext(); ) {
            CanvasModelListener l = (CanvasModelListener) it.next();
            if(e == null) {
                e = new CanvasModelEvent(this, action, oldValues, newValues);
            }
            l.modelChanged(e);
        }
    }

    public void paint(Graphics g, Selection selection) {
        Set suppressed = selection.getDrawsSuppressed();
        for(int i = 0, n = canvasObjects.size(); i < n; i++) {
            CanvasObject shape = (CanvasObject) canvasObjects.get(i);
            Graphics dup = g.create();
            if(suppressed.contains(shape)) {
                selection.drawSuppressed(dup, shape, 0, 0);
            } else {
                shape.draw(dup, 0, 0);
            }
            dup.dispose();
        }
    }

    public CanvasObject getObjectAt(int x, int y) {
        Location loc = Location.create(x, y);
        for(int i = canvasObjects.size() - 1; i >= 0; i--) {
            CanvasObject shape = (CanvasObject) canvasObjects.get(i);
            if(shape.contains(loc)) return shape;
        }
        return null;
    }
    
    public Collection getObjectsIn(Bounds bds) {
        ArrayList ret = null;
        for(int i = 0, n = canvasObjects.size(); i < n; i++) {
            CanvasObject shape = (CanvasObject) canvasObjects.get(i);
            if(bds.contains(shape.getBounds())) {
                if(ret == null) ret = new ArrayList();
                ret.add(shape);
            }
        }
        return ret == null ? Collections.EMPTY_LIST : ret;
    }
    
    public void doAction(Action action) {
        undoLog.doAction(action);
    }

    public void addObjects(Collection shapes) {
        ArrayList added = new ArrayList(shapes.size());
        for(Iterator it = shapes.iterator(); it.hasNext(); ) {
            Object value = it.next();
            if(!canvasObjects.contains(value)) {
                canvasObjects.add(value);
                added.add(value);
            }
        }
        if(!added.isEmpty()) {
            fireChanged(CanvasModelEvent.ACTION_ADDED, added, 0, 0, 0);
        }
    }
    
    public void removeObjects(Collection shapes) {
        ArrayList removed = new ArrayList(shapes.size());
        for(Iterator it = shapes.iterator(); it.hasNext(); ) {
            Object value = it.next();
            if(canvasObjects.remove(it.next())) {
                removed.add(value);
            }
        }
        if(!removed.isEmpty()) {
            fireChanged(CanvasModelEvent.ACTION_REMOVED, removed, 0, 0, 0);
        }
    }
    
    public void translateObjects(Collection shapes, int dx, int dy) {
        if(dx != 0 || dy != 0) {
            boolean found = false;
            for(Iterator it = shapes.iterator(); it.hasNext(); ) {
                DrawingMember shape = (DrawingMember) it.next();
                if(canvasObjects.contains(shape)) {
                    found = true;
                    shape.translate(dx, dy);
                }
            }
            if(found) {
                fireChanged(CanvasModelEvent.ACTION_TRANSLATED, shapes, 0, dx, dy);
            }
        }
    }

    public void moveHandle(CanvasObject shape, int index, int dx, int dy) {
        if(canvasObjects.contains(shape) && (dx != 0 || dy != 0)) {
            ((DrawingMember) shape).moveHandle(index, dx, dy);
            fireChanged(CanvasModelEvent.ACTION_HANDLE_MOVED, shape, index, dx, dy);
        }
    }

    public void insertHandle(CanvasObject shape, int index) {
        ((DrawingMember) shape).insertHandle(index);
        fireChanged(CanvasModelEvent.ACTION_HANDLE_INSERTED, shape, index, 0, 0);
    }
    
    public void deleteHandle(CanvasObject shape, int index) {
        ((DrawingMember) shape).deleteHandle(index);
        fireChanged(CanvasModelEvent.ACTION_HANDLE_DELETED, shape, index, 0, 0);
    }
    
    public void setAttributeValues(Map values) {
        HashMap oldValues = new HashMap();
        for(Iterator it = values.entrySet().iterator(); it.hasNext(); ) {
            Map.Entry entry = (Map.Entry) it.next();
            AttributeMapKey key = (AttributeMapKey) entry.getKey();
            Object value = entry.getValue();
            Attribute attr = key.getAttribute();
            DrawingMember shape = (DrawingMember) key.getObject();
            oldValues.put(key, shape.getValue(attr));
            shape.setValue(attr, value);
        }
        fireChanged(CanvasModelEvent.ACTION_ATTRIBUTE_CHANGED, oldValues, values);
    }
}
