#define _GNU_SOURCE 1
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include "main.h"

#define SIGNEX(X) (((X) & 0x8000) ? ((X) | 0xffff0000) : (X))

#define FpCond (CPR[1][31])

/* MIPS R2000/3000 Instructions */
typedef enum {SPECIAL, REGIMM, J, JAL, BEQ, BNE, BLEZ, BGTZ, ADDI, ADDIU, SLTI,
	  SLTIU, ANDI, ORI, XORI, LUI, COP0, COP1, COP2, COP3,
	  LB=32, LH, LWL, LW, LBU, LHU, LWR,
	  SB=40, SH, SWL, SW, SWR=46,
	  LWC0=48, LWC1, LWC2, LWC3,
	  SWC0=56, SWC1, SWC2, SWC3
	  } opcodefld;
typedef enum { /* SPECIAL function */
    SLL, SRL=2, SRA, SLLV, SRLV=6, SRAV, 
    JR, JALR, SYSCALL=12, BREAK,
    MFHI=16, MTHI, MFLO, MTLO,
    MULT=24, MULTU, DIV, DIVU,
    ADD=32, ADDU, SUB, SUBU, AND, OR, XOR, NOR,
    SLT=42, SLTU
    } specialfld;
enum { BLTZ, BGEZ, BLTZAL=16, BGEZAL}; /* REGIMM rt */
enum { MF=0, CF=2, MT=4, CT=6, BC=8, CO=16, FF_S=16, FF_D, FF_W=20 }; /* COPz rs */
enum { S = 0, D, DUMMY1, DUMMY2, W }; /* COPz res */
enum { BCF, BCT }; /* COPz rt */
/* MIPS R2010/3010 Floating Point Unit */
typedef enum {FADD, FSUB, FMUL, FDIV, FABS=5, FMOV, FNEG, 
	  FCVTS=32, FCVTD, FCVTW=36,
	  C_F=48, CUN, CEQ, CUEQ, COLT, CULT, COLE, CULE,
	  CSF, CNGLE, CSEQ, CNGL,  CLT, CNGE,  CLE, CNGT
	  } functionfld;
typedef enum {COND_UN=0x1, COND_EQ=0x2, COND_LT=0x4, COND_IN=0x8} fpcondcodes;

unsigned int pc, npc, nnpc, HI, LO;

void long_multiply (unsigned int v1, unsigned int v2, int sign);

void run() {

  unsigned int pc, npc, nnpc, inst, sa, cop;
  unsigned int op, rs, rt, rd, offset, target;
  double fsop, ftop, fpres = 0.0;

  npc = entry_point;
  nnpc = npc + 4;

  while (1) {

    // Make sure R[0] is always zero, even if something wrote to it previously
    R[0] = 0;

    // Increment program counter
    pc = npc;
    npc = nnpc;
    nnpc += 4;

    if (pc==0) {
      // End of program
      //printf("Finished\n");
      return;
    }

    inst = _mem_read_word(pc);

    //printf("pc=0x%08x, inst=0x%08x, npc=0x%08x, nnpc=0x%08x\n", pc, inst, npc, nnpc);

    // Dump registers
    if (opt_dumpreg)
      print_registers();

    // Print the stack
    if (opt_printstack)
      print_stack();

    // Print the instruction
    if (opt_disassemble)
      print(sim_to_real_addr(pc), 1, pc);

    // Decode inst (not all fields are used for every inst)
    op = inst >> 26;
    rs = (inst >> 21) & 0x1f;
    rt = (inst >> 16) & 0x1f;
    rd = (inst >> 11) & 0x1f;
    offset = 0xffff & inst;

	switch(op) {
	case  SPECIAL:
	    switch(inst & 0x3f) {
	    case  SLL:	sa = (inst >> 6) & 0x1f; R[rd] = R[rt] << sa; continue;
	    case  SRL:	sa = (inst >> 6) & 0x1f; R[rd] = R[rt] >> sa; continue;
	    case  SRA:
		sa = (inst >> 6) & 0x1f;
		R[rd] = (signed int) (((signed int)R[rt]) >> sa);
		continue;
	    case  SLLV:		R[rd] = R[rt] << (R[rs] & 0x1f); continue;
	    case  SRLV:		R[rd] = R[rt] >> (R[rs] & 0x1f); continue;
	    case  SRAV:
		R[rd] = (((signed int)R[rt]) >> (R[rs] & 0x1f));
		continue;
	    case  JR:   nnpc = R[rs]; continue;
	    case  JALR:	R[rd] = npc + 4; nnpc = R[rs]; continue;
	    case  SYSCALL:
                dosyscall();
                continue;
	    case  BREAK:	return;
	    case  MFHI: 	R[rd] = HI; continue;
	    case  MTHI:		HI = R[rs]; continue;
	    case  MFLO: 	R[rd] = LO; continue;
	    case  MTLO:		LO = R[rs]; continue;
	    case  MULT:		long_multiply(R[rs],R[rt], 1); continue;
	    case  MULTU:	long_multiply(R[rs],R[rt], 0); continue;
	    case  DIV:	
		LO = (signed) R[rs] / (signed) R[rt];
		HI = (signed) R[rs] % (signed) R[rt];
		continue;
	    case  DIVU:	LO = R[rs] / R[rt]; HI = R[rs] % R[rt];	continue;
	    case  ADD:
		R[rd] = (signed) R[rs] + (signed) R[rt];
		continue;
	    case  ADDU:
                R[rd] = R[rs] + R[rt]; continue;
	    case  SUB:
		R[rd] = (signed) R[rs] - (signed) R[rt]; 
		continue;
	    case  SUBU:		R[rd] = R[rs] - R[rt]; continue;
	    case  AND:		R[rd] = R[rs] & R[rt]; continue;
	    case  OR:		R[rd] = R[rs] | R[rt]; continue;
	    case  XOR:		R[rd] = R[rs] ^ R[rt]; continue;
	    case  NOR:		R[rd] = ~(R[rs]|R[rt]); continue;
	    case  SLT:
		R[rd] = ((signed) R[rs] < (signed) R[rt]) ? 1: 0; continue;
	    case  SLTU:		R[rd] = (R[rs] < R[rt]) ? 1: 0;	continue;
	    }
	case  REGIMM:
	    switch(rt) {
	    case BLTZ:
 		if((signed) R[rs] < 0)
		    nnpc = npc + (SIGNEX(offset) << 2);
		continue;
	    case BLTZAL:
		if((signed) R[rs] < 0) {
		    R[31] = npc + 4;
		    nnpc = npc + (SIGNEX(offset) << 2);
		}
		continue;
	    case BGEZ:
 		if((signed) R[rs] >= 0)
		    nnpc = npc + (SIGNEX(offset) << 2);
		continue;
	    case BGEZAL:
		if((signed) R[rs] >= 0) {
		    R[31] = npc + 4;
		    nnpc = npc + (SIGNEX(offset) << 2);
		}
		continue;
	    default:
		fprintf(stderr,"Unknown REGIMM instruction at 0x%x\n",pc);
                exit(1);
	    }
	case  J:
	    target = (npc & 0xf0000000) + ((inst & 0x03ffffff) << 2);
	    nnpc = target;
	    continue;
	case  JAL:
	    target = (npc & 0xf0000000) + ((inst & 0x03ffffff) << 2);
	    R[31] = npc + 4;
	    nnpc = target;
	    continue;
	case  BEQ:
	    if(R[rs] == R[rt])
		nnpc = npc + (SIGNEX(offset) << 2);
	    continue;
	case  BNE: 
	    if(R[rs] != R[rt])
		nnpc = npc + (SIGNEX(offset) << 2);
	    continue;
	case  BLEZ:
	    if((signed) R[rs] <= 0)
		nnpc = npc + (SIGNEX(offset) << 2);
	    continue;
	case  BGTZ:
	    if((signed) R[rs] > 0)
		nnpc = npc + (SIGNEX(offset) << 2);
	    continue;
	case  ADDI:
	    R[rt] = (signed) R[rs]+(signed)SIGNEX(offset);
	    continue;
	case  ADDIU:   R[rt] = R[rs] + SIGNEX(offset); continue;
	case  SLTI:
		R[rt] = ((signed) R[rs] < (signed) SIGNEX(offset)) ? 1 : 0;
	    continue;
	case  SLTIU:   R[rt] = (R[rs] < SIGNEX(offset)) ? 1 : 0; continue;
	case  ANDI:    R[rt] = R[rs] & offset; continue;
	case  ORI:     R[rt] = R[rs] | offset; continue;
	case  XORI:    R[rt] = R[rs] ^ offset; continue;
	case  LUI:     R[rt] = (offset << 16); continue;
	case  LB:
	    R[rt] = (signed int) ((signed char) mem_read_byte(R[rs] + SIGNEX(offset))); continue;
	case  LH:
	    R[rt] = (signed int) ((signed short) mem_read_half(R[rs] + SIGNEX(offset))); continue;
	case  LW:
	    R[rt] = (signed) mem_read_word(R[rs] + SIGNEX(offset)); continue;
	case  LBU:
	    R[rt] = mem_read_byte(R[rs] + SIGNEX(offset)); continue;
	case  LHU:
	    R[rt] = mem_read_half(R[rs] + SIGNEX(offset)); continue;
	case  SB:   mem_write_byte(R[rs] + SIGNEX(offset), R[rt]); continue;
	case  SH:   mem_write_half(R[rs] + SIGNEX(offset), R[rt]); continue;
	case  SW:   mem_write_word(R[rs] + SIGNEX(offset), R[rt]); continue;
	case  LWL: {
	    unsigned int addr = R[rs] + SIGNEX(offset);
	    unsigned int word;
	    word = mem_read_word(addr & 0xfffffffc);
	    switch(addr & 0x3) {
	    case 0: R[rt] = (word << 24) | (R[rt] & 0xffffff); continue;
	    case 1: R[rt] = (word << 16) | (R[rt] & 0xffff); continue;
	    case 2: R[rt] = (word <<  8) | (R[rt] & 0xff); continue;
	    case 3: R[rt] = word; continue;
	    }
	}
	case  LWR: {
	    unsigned int addr = R[rs] + SIGNEX(offset);
	    unsigned int word;

	    word = mem_read_word(addr & 0xfffffffc);
	    switch(addr & 0x3) {
	    case 0: R[rt] = (R[rt] & 0xffffff00) | (word >> 24); continue;
	    case 1: R[rt] = (R[rt] & 0xffff0000) | (word >> 16); continue;
	    case 2: R[rt] = (R[rt] & 0xff000000) | (word >> 8); continue;
	    case 3: R[rt] = word; continue;
	    }
	}
	case  SWL: {
	    unsigned int addr = R[rs] + SIGNEX(offset);
	    unsigned int data;

	    data = mem_read_word(addr & 0xfffffffc);
	    switch(addr & 0x3) {
	    case 0: data = (data & 0xffffff00) | (R[rt] >> 24);	break;
	    case 1: data = (data & 0xffff0000) | (R[rt] >> 16);	break;
	    case 2: data = (data & 0xff000000) | (R[rt] >> 8); break;
	    case 3: data = R[rt]; break;
	    }
	    mem_write_word(addr & 0xfffffffc, data);
	    continue;
	}
	case  SWR: {
	    unsigned int addr = R[rs] + SIGNEX(offset);
	    unsigned int data;

	    data = mem_read_word(addr & 0xfffffffc);
	    switch(addr & 0x3) {
	    case 0: data = R[rt];	break;
	    case 1: data = (R[rt] <<  8) | (data & 0xff); break;
	    case 2: data = (R[rt] << 16) | (data & 0xffff); break;
	    case 3: data = (R[rt] << 24) | (data & 0xffffff); break;
	    }
	    mem_write_word(addr & 0xfffffffc, data);
	    continue;
	}
	case  COP0: case  COP1: case  COP2: case  COP3:
	    rd = (inst >> 11) & 0x1f;
	    cop = (inst >> 26) - COP0;
	    if(cop != 1) /* Not an FPU operation */
		switch(rs) {
		case MF: R[rt] = CPR[cop][rd]; continue;
		case CF: R[rt] = CCR[cop][rd]; continue;
		case MT: CPR[cop][rd] = R[rt]; continue;
		case CT: CCR[cop][rd] = R[rt]; continue;
		case BC:
		    if(rt == BCF) {
			if(CpCond[cop] == 0)
			    nnpc = npc + (SIGNEX(offset) << 2);
			continue;
		    } else { /* BCT */
			if(CpCond[cop] != 0)
			    nnpc = npc + (SIGNEX(offset) << 2);
			continue;
		    }
		case CO:
		    fprintf(stderr,"Unknown coprocessor instruction at 0x%0x\n",pc);
                    exit(1);
		default:
		    fprintf(stderr,"Unknown coprocessor operation at 0x%0x\n",pc);
                    exit(1);
		}
	    else {
		int fmt = ((inst >> 21) & 0x1f) - 16; /* S=0, D=1, W=4 */
		int ft  = (inst >> 16) & 0x1f;
		int fs  = (inst >> 11) & 0x1f;
		int fd  = (inst >> 6) & 0x1f;
		switch(rs) {
		case MF: {
		    float val = FGR[fs];
		    unsigned int *vp = (unsigned int *) &val;
		    R[rt] = *vp;
		    continue;
		}
		case MT: {
		    unsigned int word = R[rt];
		    float *wp = (float *) &word;
		    FGR[fs] = *wp;
		    continue;
		}
		case CT:
		    CCR[1][rd] = R[rt];
		    continue;
		case CF: R[rt] = CCR[1][rd]; continue;
		case BC:
		    if(((rt == BCF) && (FpCond == 0)) ||
		       ((rt == BCT) && (FpCond != 0)))
			    nnpc = npc + (SIGNEX(offset) << 2);
			continue;
		case FF_S: case FF_D: case FF_W: /* FPU function */
		    switch(fmt) {
		    case S: fsop = FGR[fs]; ftop = FGR[ft]; break;
		    case D: fsop = FPR[fs >> 1]; ftop = FPR[ft >> 1]; break;
		    case W: fsop = FWR[fs]; ftop = FWR[ft]; break;
		    default:
                      fprintf(stderr, "Unknown format at 0x%x", pc);
                      exit(1);
		    }
		    switch(inst & 0x3f) {
		    case FABS: fpres = fabs(fsop); break;
		    case FADD: fpres = fsop + ftop; break;
		    case FSUB: fpres = fsop - ftop; break;
		    case FDIV: fpres = fsop / ftop; break;
		    case FMUL: fpres = fsop * ftop; break;
		    case FNEG: fpres = -fsop; break;
		    case FMOV: fpres = fsop; break;
		    case FCVTD: fmt = D; fpres = fsop; break;
		    case FCVTS: fmt = S; fpres = fsop; break;
		    case FCVTW: fmt = W; fpres = fsop; break;
		    case C_F:	case CUN:    case CEQ:	case CUEQ:
		    case COLT:  case CULT:   case COLE:	case CULE:
		    case CSF:	case CNGLE:  case CSEQ:	case CNGL:
		    case CLT:   case CNGE:   case CLE:	case CNGT: {
			/* BUG BUG what happened to COND_IN ?? */
			int less, equal, unordered;
			int cond = inst & 0x1f;

			if((fsop==NAN) || (ftop==NAN)) {
			    less = 0;
			    equal = 0;
			    unordered = 1;
			} else {
			    less = fsop < ftop;
			    equal = fsop == ftop;
			    unordered = 0;
			}
			FpCond = ((COND_LT&cond) ? less : 0) |
                                 ((COND_EQ&cond) ? equal: 0) |
			         ((COND_UN&cond) ? unordered: 0);
			continue; /* don't fall out to the following switch */
		    }
		    default:
			fprintf(stderr, "Unknown FPU function 0x%x at pc 0x%0x\n", mem_read_word(pc), pc);
                        exit(1);
		    }
		    switch(fmt) {
		    case S: FGR[fd] = (float) fpres; break;
		    case D: FPR[fd >> 1] = (double) fpres; break;
		    case W: FWR[fd] = (int) fpres; break;
		    /* no need for a default because they are caught earlier */
		    }
		    continue;
		default:
		    fprintf(stderr, "Unknown rs field of instruction  0x%x at pc 0x%0x\n", mem_read_word(pc), pc);
                    exit(1);
		}
	    }
	case LWC1: {
	    unsigned int *wp = (unsigned int *) &FGR[rt];
	    *wp = mem_read_word(R[rs]+SIGNEX(offset));
	    continue;
	}
        case SWC1: {
	    float val = FGR [rt];
	    unsigned int *vp = (unsigned int *) &val;
	    mem_write_word(R[rs] + SIGNEX(offset), *vp);
	    continue;
	}
	case  LWC0: case  LWC2: case LWC3:
	    CPR[(inst >> 26)-LWC0][rt] = mem_read_word(R[rs]+SIGNEX(offset));
	    continue;
	case  SWC0: case  SWC2: case SWC3:
	    mem_write_word(R[rs] + SIGNEX(offset), CPR[(inst>>26) - SWC0][rt]);
	    continue;
	default:
	    fprintf(stderr, "Unknown opcode field of instruction 0x%x at 0x%x\n", inst, pc);
            exit(1);
	}

  }
}


void long_multiply (unsigned int v1, unsigned int v2, int sign) {

  unsigned long long prod;

  if (sign) {
    // mult
    prod = (long long)(int)v1 * (int)v2;
    LO = (int)prod;
    HI = (int)(prod >> 32);

  } else {
    // multu
    prod = (unsigned long long)v1 * v2;
    LO = (int)prod;
    HI = (int)(prod >> 32);

  }

}
