#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <elf.h>

#include "main.h"

// Get half word
unsigned short gethalf(void *q) {
  unsigned short *p = q;
  return *p;
}

// Get word
unsigned int getfull(void *q) {
  unsigned int *p = q;
  return *p;
}

// Read a MIPS ELF file
void readelf(char* filename) {

  int i, j, r;
  int filesize;
  struct stat fileinfo;
  FILE* fd;
  Elf32_Ehdr* ehdr;
  void* buf;

  if (stat(filename, &fileinfo) < 0) {
    perror("Cannot open file"); exit(1);
  }

  filesize = (int)fileinfo.st_size;
#ifdef READELF_DEBUG
  printf("File size is: %d\n", filesize);
#endif

  fd = fopen(filename, "r");
  if (!fd) {
    perror("Cannot open file"); exit(1);
  }

  buf = malloc(filesize);
  if (!buf) {
    perror("malloc failed"); exit(1);
  }

  if (fread(buf, 1, filesize, fd) <= 0) {
    perror("Cannot read file"); exit(1);
  }
  fclose(fd);

  ehdr = (Elf32_Ehdr*)buf;

  if (gethalf(&ehdr->e_type) != ET_EXEC) {
    printf("file type is 0x%x, not a proper executable\n", ehdr->e_type);
    exit(1);
  }

  if (gethalf(&ehdr->e_machine) != EM_MIPS) {
    printf("machine architecture is 0x%x, not a big-endian MIPS!\n", ehdr->e_machine);
    exit(1);
  }

  entry_point = getfull(&ehdr->e_entry);

#ifdef READELF_DEBUG
  printf("program headers: %d\n", gethalf(&ehdr->e_phnum));
  printf("program header offset: 0x%08x\n", getfull(&ehdr->e_phoff));
  printf("program header entry size: 0x%08x\n", gethalf(&ehdr->e_phentsize));
  printf("section headers: %d\n", gethalf(&ehdr->e_shnum));
  printf("section header offset: 0x%08x\n", getfull(&ehdr->e_shoff));
  printf("section header entry size: 0x%08x\n", gethalf(&ehdr->e_shentsize));
#endif
  if (opt_disassemble)
    printf("Entry point is: 0x%08x\n", entry_point);

  // Skip past already allocate memory regions
  for (r=0; r<MAX_MEM_REGIONS; r++) {
    if (!memory_region[r]) break;
  }

  // Iterate over the section headers
  for (i=0; i<gethalf(&ehdr->e_shnum); i++) {
    Elf32_Shdr* shdr;
    Elf32_Shdr* strtabhdr;
    char* section_name;
    int sh_offset, sh_size, sh_addr;

    shdr = (Elf32_Shdr*) (getfull(&ehdr->e_shoff) + gethalf(&ehdr->e_shentsize) * i + buf);
    strtabhdr = (Elf32_Shdr*) (getfull(&ehdr->e_shoff) + gethalf(&ehdr->e_shentsize)
                               * gethalf(&ehdr->e_shstrndx) + buf);
    section_name = buf + getfull(&strtabhdr->sh_offset) + getfull(&shdr->sh_name);

#ifdef READELF_DEBUG
    printf("in section[%d]: %s\n", i, section_name);
#endif

    if ( !(getfull(&shdr->sh_flags) & SHF_ALLOC) ) {
      continue;
    }

    sh_offset = getfull(&shdr->sh_offset);
    sh_size = getfull(&shdr->sh_size);
    sh_addr = getfull(&shdr->sh_addr);

    memory_region[r] = malloc(sizeof(struct mem_region));
    if (!memory_region[r]) {
      perror("malloc failed"); exit(1);
    }

    memory_region[r]->addr = sh_addr;
    memory_region[r]->size = sh_size;
    memory_region[r]->ptr = calloc(sh_size, 1);
    if (!memory_region[r]->ptr) {
      perror("malloc failed"); exit(1);
    }

#ifdef READELF_DEBUG
      printf("Section offset=0x%08x, size=0x%08x, addr=0x%08x\n",
             sh_offset, sh_size, sh_addr);
#endif

    if ( getfull(&shdr->sh_type) & SHT_NOBITS ) {
      r++; // increment memory region counter
      continue;
    }

    // Copy data from the executable file into memory
    for (j=0; j<(sh_size/4); j++) {
      unsigned int* dest;
      dest = (memory_region[r]->ptr)+j*4;
      *dest = getfull(buf+sh_offset+j*4);
    }

    r++; // increment memory region counter

  }

  // Iterate over program headers
  for (i=0; i<gethalf(&ehdr->e_phnum); i++) {
    Elf32_Phdr* phdr;

    phdr = (Elf32_Phdr*) (getfull(&ehdr->e_phoff) + gethalf(&ehdr->e_phentsize) * i + buf);

    if (getfull(&phdr->p_type) == 0x70000000) {
      int foffset = 0;
      foffset = getfull(&phdr->p_offset);
      R[REG_GP] = getfull(buf + foffset + getfull(&phdr->p_filesz) - 4);
#ifdef READELF_DEBUG
      printf("Setting gp value to 0x%x\n", R[REG_GP]);
#endif
    }

  }

  free(buf);

}
